/* v0.8
 *
 * nav.c:  Ship navigation code.
 *
 * This program is free software and may be freely redistributed as
 * specified in the GNU General Public License.  Please see the file
 * 'COPYING' for details.
 */

#include "spaceconf.h"
#include "platform.h"
#include "dbint.h"
#include "space.h"
#include "object.h"
#include "nav.h"
#include "damage.h"
#include "sensors.h"
#include "smisc.h"
#include "events.h"

void navSetCourse(TAG *object, int player, float bearing, float elevation)
{
	/* check for legal values */

	if (bearing > 359 || bearing < 0) {
		Notify(player, "Valid headings are from 0 to 359 inclusive.");
		return;
	}

	if (elevation > 90 || elevation < -90) {
		Notify(player, "Valid elevation/depressions are from -90 to "
		  "90 inclusive.");
		return;
	}

	/* set desired change in heading */

	object->heading_adj.bearing = bearing - object->heading.bearing;

	if (object->heading_adj.bearing < -180)
		object->heading_adj.bearing += 360.0;

	else if (object->heading_adj.bearing > 180)
		object->heading_adj.bearing -= 360.0;

	object->heading_adj.elevation = elevation - object->heading.elevation;

	if (object->heading_adj.elevation != 0 || 
	  object->heading_adj.bearing != 0)
		FNotify(player, "Turning to %3.2f%+3.2f.", bearing, elevation);
	else
		FNotify(player, "Now heading %3.2f%+3.2f.", bearing, elevation);

	return;

}

void navAllocCheck(TAG *object)
{
	SHIP *ship = object->shipdata;
	float allowed_speed;
	dbref navigator;

	navigator = dbrefUser(ship->nav);

	if (navWarpCost(object, ship->warp_speed) > ship->alloc_nav) {

		allowed_speed = navMaxWarp(object, ship->alloc_nav);
		if (allowed_speed < 1.0)
			allowed_speed = 0.0;
		navSetWarp(object, navigator, allowed_speed);

	}
	
	return;
}

int navWarpCost(TAG *object, float speed)
{
	if (speed > 0.0 && !Speedy(object->shipdata))
		return ((int)(290 + 10 * speed * speed) * 
		  object->shipdata->warp_factor *
		  (Pokey(object->shipdata) ? 2.0 : 1.0));
	else
		return 0;	

}

float navMaxWarp(TAG *object, int cost)
{
	float warpsq;

	if (Pokey(object->shipdata))
		cost = cost / 2;	

	warpsq = ((float) cost)/(10.0 * object->shipdata->warp_factor) - 29.0;

	if (Speedy(object->shipdata))
		return object->shipdata->warp_max;
	else if (warpsq >= 1.0)
		return sqrt(warpsq);
	else
		return 0.0;
}

void navSetWarp(TAG *object, dbref player, float setting)
{

	SHIP *ship = object->shipdata;
	int energy_req;

	if (!CanMove(object)) {
		Notify(player, "This object cannot move.");
		return;
	}

	if (setting == 0.0) {
		Notify(player, "Warp disengaged.");
		ship->warp_speed = 0.0;
		return;
	}

	if (setting < 1.0) {
		Notify(player, "Warp setting must be over 1.0.");
		return;
	}

	if (setting > ship->warp_max) {
		FNotify(player, "Engines on this ship are not capable of"
		  " speeds in excess of warp %3.1f.", ship->warp_max);
		return;
	}

	energy_req = navWarpCost(object, setting);

	if (Speedy(ship))
		energy_req = 0;

	if (ship->alloc_nav < energy_req) {
		FNotify(player, "Not enough energy allocated to drives for"
		  " warp %3.1f.", setting);
		FNotify(player,"Power required for that setting is %d.",
		  energy_req);
		return;
	}

	ship->warp_speed = setting;
	FNotify(player, "Warp engaged at %3.1f.", setting);

	return;
}

void navDoorControl(TAG *object, dbref player, int action)
{
	SHIP *ship = object->shipdata;

	switch (ship->door_status) {

	  case DOORS_NONE:
		Notify(player, "This ship has no docking bay doors.");
		break;

	  case DOORS_CLOSED:
		if (!action) 
			Notify(player, "Docking bay doors are already closed.");
		else {
			ship->door_status = DOORS_OPEN;
			Notify(player, "Docking bay doors opened.");
			evBayDoors(object, 1);
		}
		break;

	  case DOORS_OPEN:
		if (action) 
			Notify(player, "Docking bay doors are already open.");
		else {
			ship->door_status = DOORS_CLOSED;
			Notify(player, "Docking bay doors closed.");
			evBayDoors(object, 0);
		}
		break;

	}

	return;
}

void navLand(TAG *object, dbref player, int target)
{
	CONTACT *planet;
	char buff[SMALL_BUF_SIZE];

	planet = snsFindContactByNumber(object, target);

	if (planet==NULL) {
		Notify(player, "Invalid target.");
		return;
	}

	if (!CanLand(object->shipdata)) {
		Notify(player, "This ship cannot land.");
		return;
	}

	if (!Planet(planet->listref)) {
		Notify(player, "You cannot land on that!");
		return;
	}

	if (distance(planet->listref->pos, object->pos) > DOCK_RANGE) {
		FNotify(player, "A ship must be within %d units of the target"
		  " to land.", (int)DOCK_RANGE);
		return;
	}

	sprintf(buff, "#%d", planet->listref->data_object);
	setAttrByName(object->data_object, STATUS_DOCKED_AT, buff);
	sprintf(buff, "%d %d %d", planet->listref->pos.x, 
	  planet->listref->pos.y, planet->listref->pos.z);
	setAttrByName(object->data_object, STATUS_STARTPOS, buff);

	evLanded(object, planet->listref);

	if (object->space == 0)
		log_space("%s (#%d) has landed on %s (#%d).",
		  object->name, object->data_object, planet->listref->name, 
		  planet->listref->data_object);

	objRemoveObject(object);

	return;

}

void navOrbit(TAG *object, dbref player, int target)
{
	CONTACT *planet;
	char buff[SMALL_BUF_SIZE];

	planet = snsFindContactByNumber(object, target);

	if (planet==NULL) {
		Notify(player, "Invalid target.");
		return;
	}

	if (!Planet(planet->listref)) {
		Notify(player, "You cannot orbit that!");
		return;
	}

	if (distance(planet->listref->pos, object->pos) > DOCK_RANGE) {
		FNotify(player, "A ship must be within %d units of the planet"
		  " to orbit it.", (int)DOCK_RANGE);
		return;
	}

	object->shipdata->warp_speed = 0.0;

	sprintf(buff, "#%d", planet->listref->data_object);
	setAttrByName(object->data_object, STATUS_DOCKED_AT, buff);
	sprintf(buff, "%d %d %d", planet->listref->pos.x, 
	  planet->listref->pos.y, planet->listref->pos.z);
	setAttrByName(object->data_object, STATUS_STARTPOS, buff);
	damWriteDamage(object->data_object, object->shipdata);

	evOrbiting(object, planet->listref);

	if (object->space == 0)
		log_space("%s (#%d) has entered orbit with %s (#%d).",
		  object->name, object->data_object, planet->listref->name, 
		  planet->listref->data_object);

	return;
}

void navLaunch(TAG *object, dbref player, dbref data)
{
	dbref planetdata;
	TAG *planet;	
	int space;

	space = atoi(getAttrByName(data, STATUS_SPACE));
	planetdata = getDbrefByName(data, STATUS_DOCKED_AT);

	if (!ValidObject(planetdata)) {
		Notify(player, "The ship is not on a legal planet.  Please"
		  " contact an admin.");
		return;
	}

	if ((planet = objFindObject(planetdata)) == NULL) {
		Notify(player, "The planet is not active.  Unable to leave.");
		return;
	}

	if (!Planet(planet)) {
		Notify(player, "The ship is not on a planet.");
		return;
	}

	if (space_lock[space] == 1) {
		Notify(player, "Space is temporarily locked.  Please try again "
		  "later.");
                return;
        }

	objAddObject(data);

	if ((object=objFindObject(data)) == NULL) {
		Notify(player, "Unable to activate ship.  Please contact an"
		  " admin.");
		return;
	}

	evLaunched(object, planet);
	return;
}

void navDock(TAG *object, dbref player, TAG *dock, int type)
{
	SHIP *dockee;
	char buff[SMALL_BUF_SIZE];
	int shield;

	if (!Ship(dock)) {
		Notify(player, "You cannot dock with that!");
		return;
	}

	dockee = dock->shipdata;

	if (dockee->door_status == DOORS_NONE)  {
		FNotify(player, "%s does not have a docking bay.", 
		  type ? "Your ship" : "The target");
		return;
	}

	if (dockee->door_size < object->size) {
		Notify(player, type ? "The tractor target is too large"
		  " to dock with you." : "Your ship is too large to dock with "
		  "that.");
		return;
	}

	if (dock->speed > 0.0 || object->speed > 0.0) {
		Notify(player, "Both your ship and the target must disengage"
		  " warp before docking may commence.");
		return;
	}

	shield = calcFacingShield(object->pos, dock);

	if (dock->shipdata->shield_status[shield] == SHLD_UP) {
		FNotify(player, "%s facing shield must be lowered before "
		  "ship can enter dock.", type ? "Your" : "Dock's");
		return;
	}

	if (distance(dock->pos, object->pos) > DOCK_RANGE) {
		FNotify(player, "A ship must be within %d units of the target"
		  " to dock.", (int)DOCK_RANGE);
		return;
	}

	if (dockee->door_status != DOORS_OPEN) {
		FNotify(player, "%s docking bay doors are closed.",
		  type ? "The ship's" : "Target");
		return;
	}

	sprintf(buff, "#%d", dock->data_object);
	setAttrByName(object->data_object, STATUS_DOCKED_AT, buff);
	sprintf(buff, "%d %d %d", dock->pos.x,	dock->pos.y, dock->pos.z);
	setAttrByName(object->data_object, STATUS_STARTPOS, buff);

	evDocked(object, dock);

	if (object->space == 0)
		log_space("%s (#%d) has docked with %s (#%d).",
		  object->name, object->data_object, dock->name, 
		  dock->data_object);

	objRemoveObject(object);

	return;

}

void navUndock(TAG *object, dbref player, dbref data)
{

	dbref dockdata;
	TAG *dock;
	int space;

	static char *shield_strings[] = {"fore", "aft", "port", "starboard"};
	space = atoi(getAttrByName(data, STATUS_SPACE));
	dockdata = getDbrefByName(data, STATUS_DOCKED_AT);

	if (!ValidObject(dockdata)) {
		Notify(player, "The ship is not in a legal dock.  Please"
		  " contact an admin.");
		return;
	}

	if ((dock = objFindObject(dockdata)) == NULL) {
		Notify(player, "The dock is not active.  Unable to leave.");
		return;
	}

	if (!Ship(dock)) {
		Notify(player, "The ship isn't currently docked.");
		return;
	}

	if (dock->shipdata->door_status != DOORS_OPEN) {
		Notify(player, "The docking bay doors are closed.  Unable "
		  "to leave.");
		return;
	}

	if (dock->shipdata->shield_status[dock->shipdata->door_side] == SHLD_UP) {
	 
		FNotify(player, "Dock's %s shield must be lowered before "
		  "ship can leave dock.", shield_strings[dock->shipdata->door_side]);
		return;
	}

	if (space_lock[space] == 1) {
		Notify(player, "Space is temporarily locked.  Please try again "
		  "later.");
                return;
        }

	objAddObject(data);

	if ((object=objFindObject(data)) == NULL) {
		Notify(player, "Unable to activate ship.  Please contact an"
		  " admin.");
		return;
	}

	evUndocked(object, dock);

	return;

}

		
				
