/* v0.8
 *
 * sensors.c:  Sensor code.
 *
 * This program is free software and may be freely redistributed as
 * specified in the GNU General Public License.  Please see the file
 * 'COPYING' for details.
 */

#include "spaceconf.h"
#include "platform.h"
#include "dbint.h"
#include "space.h"
#include "object.h"
#include "sensors.h"
#include "damage.h"
#include "smisc.h"
#include "shields.h"
#include "events.h"
#include "events.h"
#include "tactical.h"

#define Shield_string(ship, shield) \
        (ship->shield_status[shield] == SHLD_UP ? "UP  " : "DOWN")

enum contact_value {NOT_SENSED, SENSED};

/* Distance tables */
DISTENT *dist_table[NUM_SPACES]; /* Table containing all objects */
DISTENT *huge_table[NUM_SPACES]; /* Table containing only 'huge' objects */

DISTENT *new_distent(TAG *);
int check_pair(DISTENT *, DISTENT *);

void sense_huge_object(TAG *, TAG *);
TAG *nearest_huge_object(TAG *);
void new_sensor_turn(int);
void cleanup_contacts(int);
void update_contact(TAG *, TAG *, int);
CONTACT *add_contact(TAG *source, TAG *contact);
int new_contact_number(void);
void free_contact_number(int);
void new_contact(TAG *, CONTACT *);
void contact_lost(TAG *, CONTACT *, int);
void notify_contact_name(dbref, CONTACT *);

/* Sensor routines */

/*
 * snsBuildDistanceTables:  Build a list of tags ordered by object's range 
 *			   from hub.
 */
void snsBuildDistanceTables(int space)
{

	DISTENT *newent, *htail=NULL, *dptr, *dprev;
	TAG *tag;

	/* Cycle through the space list */	
	for (tag=space_list[space]; tag != NULL; tag=tag->next) {

		if (Removed(tag))
			continue;

		if (Huge(tag)) {

			/* Allocate new distance table entry */
			if ((newent = new_distent(tag))==NULL) {
				log_space("Out of virtual memory:  Can't "
				  "build huge distance table.");
				return;
			}

			if (htail==NULL)
				htail = huge_table[space] = newent;
			else {
				newent->prev = htail;
				htail->next = newent;
				htail = newent;
			}
		}

		/*
		 * Add entry regardless of size to range-ordered distance 
		 * table.
		 */

		/* Allocate new distance table entry */
		if ((newent = new_distent(tag))==NULL) {
			log_space("Out of virtual memory:  Can't "
			  "build distance table.");
			return;
		}

		dprev = NULL;

		for (dptr=dist_table[space]; dptr != NULL; dptr=dptr->next) {

			if (newent->range < dptr->range)
				break;

			dprev = dptr;
		}

		if (dprev==NULL) {

			dist_table[space] = newent;
			newent->next = dptr;

		} else {

			dprev->next = newent;
			newent->prev = dprev;
			newent->next = dptr;
		}
					
		if (dptr != NULL)
			dptr->prev = newent;
		
	}				
			
	return;				
	
}

DISTENT *new_distent(TAG *object)
{
	DISTENT *newent;

	/* Allocate new distance table entry */
	if ((newent = (DISTENT *) malloc(sizeof(DISTENT))) == NULL) {
		return NULL;
	}
	newent->object = object;
	newent->range = HubRange(object->pos);
/*	newent->width = 0; */
	newent->prev = newent->next = NULL;

	return (newent);

}
	
/*
 * snsFreeDistanceTables: Delete the current free distance tables
 */

void snsFreeDistanceTables(int space)
{

	DISTENT *ptr, *tmpptr;

	ptr = dist_table[space];
	while (ptr != NULL) {
		tmpptr=ptr;
		ptr=ptr->next;
		free(tmpptr);
	}
	dist_table[space] = NULL;
		
	ptr = huge_table[space];
	while (ptr != NULL) {
		tmpptr=ptr;
		ptr=ptr->next;
		free(tmpptr);
	}
	huge_table[space] = NULL;

	return;
}

void snsCheckSensors(int space)
{

	DISTENT *entry, *ptr;

	/* Reset contacts state for new turn. */
	new_sensor_turn(space);

	for (entry=dist_table[space]; entry != NULL; entry=entry->next) {

		if (Removed(entry->object))
			continue;

		if (CanSense(entry->object)) {

			/* Check objects forwards in the list */	
	    		for (ptr=entry->next; ptr != NULL; ptr=ptr->next) 
				if (check_pair(entry, ptr))
					break;
	
			/* Check objects backwards in the list */	
			for (ptr=entry->prev; ptr != NULL; ptr=ptr->prev) 
				if (check_pair(entry, ptr))
					break;

			for (ptr=huge_table[space]; ptr != NULL; ptr=ptr->next)
				if (entry->object != ptr->object)
					sense_huge_object(entry->object, 
					  ptr->object);	

		}

	}

	/* Remove stale contacts */
	cleanup_contacts(space);

	return;

}

/* check_pair:  Ensure that a full sensor check is only done if sensee is
 *		  within sensor's sensor range.
 */
int check_pair(DISTENT *senser, DISTENT *sensee)
{

	if (Invisible(sensee->object)) {
		update_contact(senser->object, sensee->object, NOT_SENSED);
		return 0;
	}

	if (Omniscient(senser->object)) {
		update_contact(senser->object, sensee->object, SENSED);
		return 0;
	}

	if (Huge(sensee->object)) 
		/* Skip huge objects (for now) */
		return 0;

	else {			

		/* See if sensee is even remotely in range */

		if ((abs(sensee->range - senser->range) / 1.5) <= 
		  senser->object->sensor_range) {
			snsSenseObject(senser->object, sensee->object);
			return 0;
		}
		else 
			/* Done looking in this direction */
			return 1;

	}
		

	return 1;
}

/* 
 * sense_huge_object:  Use simple formula to see if huge object appears
 * 			on sensors.
 */
void sense_huge_object(TAG *senser, TAG *sensee)
{

	int result;
	int range;

	range = distance(senser->pos, sensee->pos);

	if ((range < (senser->sensor_range * sensee->size) ||
	  Omniscient(senser)) && !Invisible(sensee))
		result = SENSED;	
	else
		result = NOT_SENSED;

	update_contact(senser, sensee, result);

	return;
}			

/*
 * snsSenseObject:  Use more complex formula to see if ship-sized object
 *		    appears on sensors.
 */	
void snsSenseObject(TAG *senser, TAG *sensee)
{

	float effective_aspect_ratio; 	/* ease of sensing target */
	float prob;
	float planet_effect, weapons_effect;
	float modified_cloak_eff;
	TAG *hugeobj;
	int i;
	int hugerange, range;

	range = distance(senser->pos, sensee->pos);

	/*
	 * If sensee is over three times the sensor range away from the senser,
	 * don't bother doing any further calculations (to save time).
	 */
	if (range > (3.0 * senser->sensor_range)) {
		update_contact(senser, sensee, NOT_SENSED);
		return;
	}

	if (Nearsighted(senser)) {
		if (range < senser->sensor_range)
			update_contact(senser, sensee, SENSED);
		else
			update_contact(senser, sensee, NOT_SENSED);
	}


	/*
	 * Add cloak effect 
	 */

	modified_cloak_eff = 1.0;
	if (Cloaked(sensee)) {
		modified_cloak_eff = sensee->cloak_effect * 10.0;
	
		if (Ship(sensee)) {
			modified_cloak_eff += 10.0 * (100.0 - 
			  (float) sensee->shipdata->reactor_setting) /
			  (float) sensee->shipdata->reactor_setting;
		}

	}

	effective_aspect_ratio = sensee->size / modified_cloak_eff;

	/* 
	 * Planet effect makes uncloaked ships harder to see, but makes
	 * cloaked ships easier to see.  Ignore if less than 5%, however.
	 */

	hugeobj = nearest_huge_object(senser);

	if (hugeobj == NULL)
		planet_effect = 0;
	else {
		hugerange= distance(sensee->pos, hugeobj->pos);
		planet_effect = hugeobj->size * 0.80 / (hugeobj->size +
			(hugerange * hugerange / 1000000.0));
	}

	if (planet_effect >= 0.05) {

		if (Cloaked(sensee)) 
			effective_aspect_ratio *= (1.0 + (2 * planet_effect));
		else
			effective_aspect_ratio *= (1.0 - planet_effect);
	}

	if (Ship(sensee)) {

	/*
	 * Weapons effect makes cloaked ships a lot easier to see, and
 	 * makes seeing uncloaked ships somewhat easier.
	 */

		weapons_effect = (((float)(sensee->shipdata->num_torps_online)
		  *TORP_CHARGING_PENALTY) + 
		  ((float)(sensee->shipdata->num_guns_online) *
		  GUN_ONLINE_PENALTY));

		/* count armed torps */
		for (i=0; i < sensee->shipdata->number_of_torps; i++)
			if (sensee->shipdata->torp[i].status == TORP_ARMED)
				weapons_effect += TORP_ARMED_PENALTY;
	}
	else 
		weapons_effect = 0.0;

	if (Cloaked(sensee))
		effective_aspect_ratio += weapons_effect;
	else
		effective_aspect_ratio += weapons_effect * 0.25;

	if (Hazy(sensee))
		effective_aspect_ratio *= 0.3;

	if (HasCataracts(senser))
		effective_aspect_ratio *= 0.3;

	/* If sensee is locked on sensor, he's really easy to see. */
	if (sensee->locked_on != NULL)
		if (sensee->locked_on->listref == senser)
			effective_aspect_ratio += 10.0;

	if (sensee->pending_lock != NULL)
		if (sensee->pending_lock->listref == senser)
			effective_aspect_ratio += 10.0;

	/* 
	 * If sensee has engaged a tractor beam on the senser, or vice-versa
	 * he's also really easy to see.
	 */

	if (Ship(senser) && senser->shipdata->tractor_target == sensee)
		effective_aspect_ratio += 10.0;

	if (senser->tractor_source == sensee)
		effective_aspect_ratio += 10.0;

	prob = 1.0 - range / (senser->sensor_range * effective_aspect_ratio);

	/* Warp speed adjustment */

	prob += ((sensee->speed - senser->speed) / 50.0);

	/* Bonus if senser has already made contact with sensee */
	if (snsFindContact(senser, sensee) != NULL)
		prob += 0.25;

	if (senser->locked_on != NULL)
		if (senser->locked_on->listref == sensee)
			prob += 0.25;

	if (FRAND < prob)
		update_contact(senser, sensee, SENSED);
	else
		update_contact(senser, sensee, NOT_SENSED);

	return;
}

/*
 * nearest_huge_object:   Find the nearest 'huge' object.
 */
TAG *nearest_huge_object(TAG *senser)			
{
	TAG *nearobj = NULL;
	int nearrange, range;
	CONTACT *cptr;

	nearrange = 0;

	for (cptr=senser->contact_list;cptr != NULL; cptr=cptr->next) {
		if (!Huge(cptr->listref))
			continue;
		range = distance(senser->pos, cptr->listref->pos);
		if (range < nearrange || nearobj==NULL) {
			nearobj = cptr->listref;
			nearrange = range;
		}
	}

	return nearobj;
}

/* Contact handling routines */

/*
 * new_sensor_turn:  Set all the 'updated' flags to FALSE on each contact
 */
void new_sensor_turn(int space)
{
	TAG *ptr;
	CONTACT *cptr;

	for (ptr=space_list[space]; ptr != NULL; ptr=ptr->next) {

		for (cptr=ptr->contact_list; cptr != NULL; cptr = cptr->next)
			cptr->updated = FALSE;

	}

	return;
}

/* cleanup_contacts:  Remove contacts that haven't been updated this turn.
 *			Assume that they no longer exist. 
 */
void cleanup_contacts(int space)
{

	TAG *ptr;
	CONTACT *cptr, *cprev, *tmpptr;

	for (ptr=space_list[space]; ptr != NULL; ptr=ptr->next) {

		cprev = NULL;
		cptr = ptr->contact_list;

		while (cptr != NULL) {

			if (cptr->updated == FALSE) {

				contact_lost(ptr, cptr, 0);
				if (cprev == NULL) 
					ptr->contact_list = cptr->next;
				else
					cprev->next = cptr->next;
				tmpptr = cptr;
				cptr = cptr->next;
				free_contact_number(tmpptr->contact_number);
				free(tmpptr);
			}
			else {	
				cprev = cptr;
				cptr = cptr->next;
			}
		}

	}

	return;

}

/* snsRemoveContact:  Remove sensee.  */
void snsRemoveContact(TAG *senser, TAG *sensee, int silent)
{

	CONTACT *cprev = NULL;
	CONTACT *cptr = senser->contact_list;
	CONTACT *tmpptr;

	while (cptr != NULL) {

		if (cptr->listref == sensee) {

			contact_lost(senser, cptr, silent);

			if (cprev == NULL)
				senser->contact_list = cptr->next;
			else
				cprev->next = cptr->next;

			tmpptr = cptr;
			cptr = cptr->next;
			free_contact_number(tmpptr->contact_number);
			free(tmpptr);
		}
		else {	
			cprev = cptr;
			cptr = cptr->next;
		}

	}

	return;

}
	
void update_contact(TAG *senser, TAG *sensee, int action)
{
	CONTACT *contact;
	float prob, roll;
	int new_info_level, is_new_contact = 0;

	contact = snsFindContact(senser, sensee);
	
	if (action == SENSED) {

		if (contact == NULL) {
			contact = add_contact(senser, sensee);
			is_new_contact = 1;
		}

		contact->turns_of_contact++;
		contact->turns_since_last_contact = 0;
		contact->last_pos.x = contact->listref->pos.x;
		contact->last_pos.y = contact->listref->pos.y;
		contact->last_pos.z = contact->listref->pos.z;
		contact->updated=TRUE;	

		if (contact->info_level == 5)
			return;

		/* See how much information we can get. */
		prob = (float)(contact->turns_of_contact) / 20.0;
		roll = FRAND;

		if (Omniscient(senser) || Nearsighted(senser) ||
			Huge(sensee))
			roll = 0;
			
		if (roll < prob)
			new_info_level = 5;
		else if (roll < (prob + .10))
			new_info_level = 4;	
		else if (roll < (prob + .20))
			new_info_level = 3;	
		else if (roll < (prob + .30))
			new_info_level = 2;
		else
			new_info_level = 1;

		/* Never show full info on cloakers */	
		if (Cloaked(contact->listref) && new_info_level > 3)
			new_info_level = 3;

		if (is_new_contact)
			new_contact(senser, contact);

		if (new_info_level > contact->info_level) {
			contact->info_level = new_info_level;
			if (!is_new_contact && Ship(senser))
				snsDisplaySensorInfo(senser, contact, -1, DSD_UPDATE);
		}


	} else {

		if (contact == NULL)
			return;			

		contact->turns_since_last_contact++;

		if (!Cloaked(contact->listref) && !Invisible(contact->listref)
		  && !Huge(contact->listref) && 
 		  contact->turns_since_last_contact < 3)
			contact->updated=TRUE;

	}

	return;
}
	
		
/*
 * add_contact:  Add contact to contact list
 */
CONTACT *add_contact(TAG *source, TAG *contact)
{

	CONTACT *ptr;
	CONTACT *newcontact;

	if ((newcontact = (CONTACT *) malloc(sizeof(CONTACT))) == NULL)
		return(NULL);

	newcontact->listref = contact;
	newcontact->contact_number = new_contact_number();
	newcontact->turns_of_contact = 0;
	newcontact->turns_since_last_contact = 0;
	newcontact->info_level = 0;
	newcontact->last_pos.x = 0;
	newcontact->last_pos.y = 0;
	newcontact->last_pos.z = 0;
	newcontact->watcher = 0;
	newcontact->inside_critical = FALSE;
	newcontact->next = NULL;

	if (source->contact_list==NULL)
		source->contact_list = newcontact;
	else {
		for (ptr=source->contact_list; ptr->next != NULL; ptr=ptr->next)			;
		ptr->next = newcontact;
	}

	return(newcontact);
}

/*
 * snsFindContact:  Find contact in contact list
 */
CONTACT *snsFindContact(TAG *source, TAG *contact)
{
	CONTACT *ptr = NULL;

	for (ptr=source->contact_list; ptr != NULL; ptr=ptr->next) 
		if (ptr->listref==contact)
			break;

	return(ptr);
}

CONTACT *snsFindContactByNumber(TAG *source, int num)
{
	CONTACT *ptr = NULL;

	for (ptr=source->contact_list; ptr != NULL; ptr=ptr->next) 
		if (ptr->contact_number == num)
			break;

	return(ptr);
}

void new_contact(TAG *senser, CONTACT *contact) {

	if (Ship(senser))
		snsDisplaySensorInfo(senser, contact, -1, DSD_NEW);

	if (EventDriven(senser)) 
		evNewContact(senser, contact);	

#ifdef LOG_CONTACTS
	if (senser->space == 0)
		log_space("NEW CONTACT!  %s (#%d) sees %s (#%d) at range %d.",
		  senser->name, senser->data_object, contact->listref->name, 
		  contact->listref->data_object,
		  distance(senser->pos, contact->listref->pos));
#endif

}
		
/* Contact number assignment routines */

#define MAX_CONTACTS 2000

enum number_status { NOT_TAKEN, TAKEN };
int contact_numbers[MAX_CONTACTS];

/*
 * new_contact_number:  Find a valid new contact number
 */
int new_contact_number(void)
{
	int i;
	static int first_time = TRUE;
	static int last_new_contact;
	static unsigned int last_outside_number = MAX_CONTACTS;
	
	if (first_time == TRUE) {

		/* Initialize contact number flags */
		for (i=0; i < MAX_CONTACTS; i++)
			contact_numbers[i] = NOT_TAKEN;

		first_time = FALSE;

	}

	/* Start looking at the next number after the last one picked. */
	for (i=last_new_contact + 1; i < MAX_CONTACTS; i++) {
		if (contact_numbers[i] == NOT_TAKEN) {
			contact_numbers[i] = TAKEN;
			last_new_contact = i;
			return(i+1);
		}
	}

	/* None found?  OK, start looking at the beginning of the list. */
	for (i=0; i <= last_new_contact; i++) {
		if (contact_numbers[i] == NOT_TAKEN) {
			contact_numbers[i] = TAKEN;
			last_new_contact = i;
			return(i+1);
		}
	}

	/* 
	 * Looks like we're going to have to take a number from outside the
	 * MAX_CONTACTS range.  Ugh.  This always has a chance of 
	 * overflowing.
	 */

	return(++last_outside_number);

}

void free_contact_number(int num) {

	if (num	> MAX_CONTACTS)
		return;
	else 
		contact_numbers[num-1] = NOT_TAKEN;

	return;
}

/*
 * snsListContacts:  Spit contact list into buff, as either contact #'s 
 *		     (format = 0), or dbrefs (format = 1).  Show the entire
 *		     list (type=0), objects that can be beamed to (type=1),
 *		     and objects that can be beamed (type=2).
 */
void snsListContacts(char *buff, CONTACT *ptr, int type, int format)
{
	static char *buffstart = NULL;
	
	if (buffstart == NULL) {
		buffstart = buff;
		buff[0]='\0';
	}

	if (buff - buffstart >= LARGE_BUF_SIZE - 10 || ptr == NULL) {
		if (buff[strlen(buff) - 1] == ' ')
			buff[strlen(buff) - 1] = '\0';
		buffstart = NULL;
		return;
	}

	if (type==0 || ((type==1) && CanBeamTo(ptr->listref)) || ((type==2) &&
		Beamable(ptr->listref))) {

		if (format)
			sprintf(buff, "#%d ", ptr->listref->data_object);
		else
			sprintf(buff, "%d ", ptr->contact_number);

	}

	snsListContacts(buff + strlen(buff), ptr->next, type, format);

	buffstart = NULL;
	return;
}

void snsDisplayContacts(TAG *object, dbref player, int show_ships,
  int show_other)
{
	CONTACT *contact;
	int found = 0;

	for (contact=object->contact_list; contact != NULL; contact=contact->next) {
		if (Ship(contact->listref) && !show_ships) continue;
		if (!Ship(contact->listref) && !show_other) continue;

		snsDisplaySensorInfo(object, contact, player, DSD_ROUTINE);
		found = 1;

	}

	if (!found)
		Notify(player, "No contacts on sensors.");

	return;
}

void snsDisplaySensorInfo(TAG *senser, CONTACT *contact, dbref player, 
  int control)
{
	char buff[SMALL_BUF_SIZE];
	XYZ xyz;
	SPH sph;
	TAG *target;

	const char *shield_strings[] = {"fore", "aft", "port", "starboard"};

	/* calculate relative positions */
	xyz.x = contact->listref->pos.x - senser->pos.x;
	xyz.y = contact->listref->pos.y - senser->pos.y;
	xyz.z = contact->listref->pos.z - senser->pos.z;
	xyz_to_sph(xyz, &sph);

	/* if player = -1, find the dbref of the tacofficer */
	if (player == -1)
		player=dbrefUser(senser->shipdata->tactical);

	target = contact->listref;

	/* display header message */
	switch (control) {
	  case DSD_NEW:
		FNotify(player, "NEW CONTACT - designated number %d",
			contact->contact_number);
		break;
	  case DSD_ROUTINE:
		sprintf(buff, "Contact [%d]:", contact->contact_number);

		if (contact->listref->locked_on != NULL) {
			if (contact->listref->locked_on->listref == senser) 
				sprintf(buff, "[*] Contact [%d]:", 
				  contact->contact_number);
		}
		Notify(player, buff);

		break;
	  case DSD_UPDATE:
		FNotify(player, "Update:  Contact [%d] further identified as:",
		  contact->contact_number);
		break;
	}
	
	notify_contact_name(player, contact);

	FNotify(player, "   Contact bearing %3.2f elevation %+2.2f range %d",
	  sph.bearing, sph.elevation, sph.range);

	if (contact->info_level > 1)  {

		if (Ship(target) || target->speed > 0) {
			/* construct the target's movement string */
			FNotify(player, "   Contact heading %3.2f%+2.2f at "
			  "warp %3.1f", target->heading.bearing, 
			  target->heading.elevation, 
			  target->speed);
		}

		if (Ship(target)) 
			/* and give the shield facings */
			FNotify(player, "   Our %s side is facing their %s "
			  "side.", shield_strings[calcFacingShield(target->pos, 
			  senser)], shield_strings[calcFacingShield(senser->pos,
			  target)]);
		else 
			FNotify(player, "   Our %s side is facing the contact.",
			  shield_strings[calcFacingShield(target->pos,senser)]);
	}

	if (Cloaked(target))
		Notify(player, "Contact is currently cloaked.");		

	Notify(player, " ");
			
	return;

}

void notify_contact_name(dbref player, CONTACT *contact)
{
 	char name[MAX_ATTRIBUTE_LEN], buff[SMALL_BUF_SIZE];
	char *fargs[1];
	SHIP *ship;
	TAG *target;

	target = contact->listref;
	ship = target->shipdata;

	fargs[0]=AllocBuffer("evDisplay");
	sprintf(fargs[0], "%d", contact->info_level);
	strcpy(name, getEvalAttr(target->data_object, INFO_CONTACT_STRING, 
	  fargs, 1));

	if (strlen(name)==0)
		strcpy(name, target->name);

	if (!Ship(target))
		Notify(player, name);

	else {
		switch (contact->info_level) {

		  case 5:
			sprintf(buff, "%s -- %s %s-class %s", target->name,
			  ship->owner_name, ship->class, ship->type);
	
			break;
		  case 4:
			sprintf(buff, "%s %s-class %s", ship->owner_name,
			  ship->class, ship->type);
			break;
		  case 3:
			sprintf(buff, "%s %s", ship->owner_name, 
			  ship->type);
			break;
		  default:
			sprintf(buff, "%s", ship->type);
			break;
	
		}

		Notify(player, buff);

	}
}


void snsScan(TAG *object, dbref player, int num)
{

	XYZ xyz;
	SPH sph;
	SHIP *scannee;
	CONTACT *source;
	CONTACT *target;
	char buff[MAX_ATTRIBUTE_LEN];
	char *fargs[1];

	const char *shield_strings[] = {"fore", "aft", "port", "starboard"};

	if (object->shipdata->damage[SYSTEM_SCANNERS].status > '6') {
		Notify(player, "Scanners are damaged and inoperable.");
		return;
	}

	target = snsFindContactByNumber(object, num);

	if (target==NULL) {
		Notify(player, "Invalid target.");
		return;
	}

	if (distance(object->pos, target->listref->pos) > 
	  object->shipdata->scanner_range) {
		Notify(player, "That object is out of scanner range.");
		return;
	}

	/* Raise our info level on the target object. */

	if (Cloaked(target->listref))
		target->info_level = Max(target->info_level, 4);
	else
		target->info_level = 5;


	FNotify(player, "Scanning [%d]: %s", target->contact_number,
	  target->listref->name);

	xyz.x = target->listref->pos.x - object->pos.x;
	xyz.y = target->listref->pos.y - object->pos.y;
	xyz.z = target->listref->pos.z - object->pos.z;
	xyz_to_sph(xyz, &sph);

	if (!Ship(target->listref)) {
		fargs[0]=AllocBuffer("snsScan");
		sprintf(fargs[0], "%d", target->contact_number);
		Notify(player, getEvalAttr(object->data_object, 
	  	  INFO_SCAN_STRING, fargs, 1));
		return;
	}
	
	if (!Ship(target->listref))
		return;

	scannee = target->listref->shipdata;

    strcpy(buff, getAttrByNumber(target->listref->data_object,
      A_DESC));
    
    if (strlen(buff))
    		Notify(player, buff);

	FNotify(player, "Target bearing %3.2f%+2.2f at range %d",
	  sph.bearing, sph.elevation, sph.range);

	FNotify(player, "       heading %3.2f%+2.2f at warp %1.0f",
	  target->listref->heading.bearing, target->listref->heading.elevation,
	  target->listref->speed);	

	/* weapons status */

	Notify(player, "Weapons:");
	FNotify(player, "    %d %ss online -- energy allocated %d",
	  scannee->num_guns_online, scannee->gun_string,
	  scannee->talloc_guns);
	FNotify(player, "    %d %ss online -- energy allocated %d",
	  scannee->num_torps_online, scannee->torp_string,
	  scannee->talloc_torps);
	Notify(player, " ");

	if (Cloaked(target->listref))
		Notify(player, "Target's cloaking device is engaged.");
	else {
		Notify(player, "Shields:");
		FNotify(player, "Fore  %d:%s  Aft  %d:%s  Port  %d:%s"
		  "  Starboard:  %d:%s",
		  scannee->shield_level[0], Shield_string(scannee, 0),
		  scannee->shield_level[1], Shield_string(scannee, 1),
		  scannee->shield_level[2], Shield_string(scannee, 2),
		  scannee->shield_level[3], Shield_string(scannee, 3));
	}

	FNotify(player, "We are currently facing their %s shield.",
	  shield_strings[calcFacingShield(object->pos, target->listref)]);

	FNotify(player, "They are currently facing our %s shield.",
	  shield_strings[calcFacingShield(target->listref->pos, object)]);

	Notify(player, " ");

	if (scannee->door_status != DOORS_NONE) {
		FNotify(player, "The target's cargo bay doors are %s.",
		  scannee->door_status == DOORS_CLOSED ? "closed" : "open");

		Notify(player, " ");
	}

	FNotify(player, "Hull integrity:  %2.0f%%\tpower to warp: %d",
	  (float)scannee->current_integrity / (float)scannee->hull_integrity
	  * 100.0, scannee->alloc_nav);

	source = snsFindContact(target->listref, object);

	if (source != NULL) 
		FNotify(dbrefUser(scannee->tactical), 
		  "We are being scanned by contact [%d]: %s.",
		  source->contact_number, object->name);
	else
		Notify(dbrefUser(scannee->tactical), 
		  "We are being scanned by an unknown source.");

	return;

}

void contact_lost(TAG *senser, CONTACT *contact, int silent)
{

	XYZ last_position;
	SPH last_relative;
	dbref tacofficer;

	if (Ship(senser) && !silent) {
		if (senser->shipdata->hull_integrity < 0)
			return;

		tacofficer = dbrefUser(senser->shipdata->tactical);

		FNotify(tacofficer, "CONTACT [%d] LOST:",contact->contact_number);
		notify_contact_name(tacofficer, contact);

		last_position.x = contact->last_pos.x - senser->pos.x;
		last_position.y = contact->last_pos.y - senser->pos.y;
		last_position.z = contact->last_pos.z - senser->pos.z;

		xyz_to_sph(last_position, &last_relative);
		FNotify(tacofficer, "Contact last spotted bearing: "
			"%3.0f%+2.0f, range %d", last_relative.bearing,
			last_relative.elevation, last_relative.range);

		if (contact->info_level > 1) 
			FNotify(tacofficer, "                     heading: %3.0f"
			  "%+2.0f at warp %3.1f", 
			  contact->listref->heading.bearing,
			  contact->listref->heading.elevation, 
			  contact->listref->speed);

	}

	if (senser->locked_on == contact)
		tacBreakLock(senser);

	if (Ship(senser))
		if (senser->shipdata->tractor_target == contact->listref)
			tacBreakTractor(senser);

	/*
	 * If the lock is broken before it is achieved, we can simulate a
	 * message to send to the tacofficer, by granting the lock then breaking
	 * it.
	 */

	if (senser->pending_lock == contact) {
		senser->locked_on = contact;
		tacBreakLock(senser);
		senser->pending_lock = NULL;
	}
		
	if (EventDriven(senser) && !silent)  {

		if (contact->inside_critical)
			evOutsideCritical(senser, contact);

		evContactLost(senser, contact);
	}

#ifdef LOG_CONTACTS
	if (senser->space == 0 && !silent)
		log_space("CONTACT LOST.  %s (#%d) no longer sees %s (#%d), "
		  "range %d.", senser->name, senser->data_object, 
		  contact->listref->name, contact->listref->data_object,
		  distance(senser->pos, contact->listref->pos));
#endif

	return;
}
