/* v0.8
 *
 * smain.c:  Main space loop.
 *
 * This program is free software and may be freely redistributed as
 * specified in the GNU General Public License.  Please see the file 
 * 'COPYING' for details.
 */

#include "spaceconf.h"
#include "platform.h"
#include "dbint.h"
#include "space.h"
#include "object.h"
#include "nav.h"
#include "shields.h"
#include "tactical.h"
#include "eng.h"
#include "damage.h"
#include "smisc.h"
#include "events.h"
#include "version.h"
#include "commands.h"
#include "sensors.h"

/*
 * Global linked list pointers.
 */
TAG *space_list[NUM_SPACES] = { NULL, NULL };

char *space_names[] = { "real", "sim" };

char *system_names[] = { "fore shield", "aft shield", "port shield",
   "starboard shield", "gun 0", "gun 1", "gun 2", "gun 3", "gun 4", "gun 5",
   "torp 0", "torp 1", "torp 2", "torp 3", "torp 4", "torp 5",
   "batteries", "cloak", "" /* unused */, "engine", "" /* unused */, "sensors", 
   "scanners", "hull" , "tractor beam", "transporters" };

int space_lock[NUM_SPACES] = {0, 0};

void move_objects(int);
void tactical_turn(int);
void handle_tractor_target(TAG *, TAG *);
void handle_tractor_source(TAG *, TAG *);

void spaceVersion(dbref player)
{
	Notify(player, ENGINE_VERSION);
	return;
}

void spaceInit(void)
{
	cmdInitFunctionTable();
	objInitSpace();

	return;
}

void spaceUpdate(void)
{
	static int count = 0;
	int current_space;

	for (current_space=0; current_space < NUM_SPACES; current_space++) {

		switch (count) {

		  case 0:
			tactical_turn(current_space);
			break;

		  case 1:
			move_objects(current_space);
			break;

		  case 2:
			snsFreeDistanceTables(current_space);
			snsBuildDistanceTables(current_space);
			snsCheckSensors(current_space);
			evCheckEvents(current_space);
			break;

		  case 3:
			move_objects(current_space);
			break;

		  case 4:
			damShipRepairs(current_space);
			break;

		  case 5:
			move_objects(current_space);
			break;

		}

		objFreeMarkedObjects(current_space);
	}

	count = ++count % 6;

	return;

}

void move_objects(int space)
{
	TAG *object;
	float inc, einc, binc, max_speed;
	SPH sph;
	XYZ xyz;

	for (object=space_list[space]; object != NULL; object=object->next) {

		if (Removed(object))
			continue;

		/*
		 * Handle tractor beams.  Ugh, so much for the nice concise
		 * movement code.  Ah, the cost of innovation!
		 */
		if (object->tractor_source != NULL && Ship(object)) 
			handle_tractor_target(object->tractor_source, object);

		if (Ship(object))
		  if (object->shipdata->tractor_target != NULL) 
			handle_tractor_source(object, 
			  object->shipdata->tractor_target);


		if (!CanMove(object))
			continue;

		if (ManualPosition(object)) {
			object->pos.x = atoi(getEvalAttr(object->data_object, 
			  MOVE_POS_X, (char **)NULL, 0));
			object->pos.y = atoi(getEvalAttr(object->data_object, 
			  MOVE_POS_Y, (char **)NULL, 0));
			object->pos.z = atoi(getEvalAttr(object->data_object, 
			  MOVE_POS_Z, (char **)NULL, 0));
			return;
		}

		else if (ManualDirection(object)) {
			object->heading.bearing = 
			  atof(getEvalAttr(object->data_object, 
			  MOVE_HEAD_BEARING, (char **)NULL, 0));
			object->heading.elevation = 
			  atof(getEvalAttr(object->data_object,
			  MOVE_HEAD_ELEVATION, (char **)NULL, 0));
			object->heading.range = 1;
			object->speed = atof(getEvalAttr(object->data_object, 
			  MOVE_SPEED, (char **)NULL, 0));
		}
		else if (Ship(object)) {

			if (Pokey(object->shipdata)) {
				max_speed = navMaxWarp(object, object->shipdata->alloc_nav);

				if (max_speed < object->shipdata->warp_speed) {
					object->shipdata->warp_speed = max_speed;
					FNotify(dbrefUser(object->shipdata->nav),
					  "Warp speed reduced to %3.2f.", max_speed);
				}
			}

			object->speed = object->shipdata->warp_speed;

			/* Adjust heading */
			if (object->heading_adj.bearing != 0 ||
				object->heading_adj.elevation != 0) {

				inc = 90.0 / (object->size * object->size);
				binc = Min(fabs(object->heading_adj.bearing),
				  inc) * Sign(object->heading_adj.bearing);
				einc = Min(fabs(object->heading_adj.elevation),
				  inc) * Sign(object->heading_adj.elevation);

				object->heading.bearing += binc;

				if (object->heading.bearing >= 360.0)
					object->heading.bearing -= 360.0;
				if (object->heading.bearing < 0)
					object->heading.bearing += 360.0;
			
				object->heading_adj.bearing -= binc;
				object->heading.elevation += einc;
				object->heading_adj.elevation -= einc;
				object->heading.range = 1;

				if (object->heading_adj.bearing==0 &&
				  object->heading_adj.elevation==0)
				  FNotify(dbrefUser(object->shipdata->nav),
				    "Now heading %3.2f%+3.2f.", 
				    object->heading.bearing, 
				    object->heading.elevation);

			}
		}

		if (object->speed < 1.0)
			continue;

		sph.bearing = object->heading.bearing;
		sph.elevation = object->heading.elevation;
		sph.range = object->speed * object->speed * object->speed;

		sph_to_xyz(sph, &xyz);

		object->pos.x += xyz.x;
		object->pos.y += xyz.y;
		object->pos.z += xyz.z;
	}

	return;
}

void tactical_turn(int space)
{

	TAG *ptr;

	for (ptr = space_list[space]; ptr != NULL; ptr=ptr->next) {

		if (Removed(ptr))
			continue;

		if (Ship(ptr)) {
			engTacticalTurn(ptr);
			tacMaintainWeapons(ptr);
			shdMaintainShields(ptr);
		}

		/* resolve any pending lock */
		if (ptr->pending_lock != NULL) {
			ptr->locked_on = ptr->pending_lock;
			ptr->pending_lock = NULL;

			if (Ship(ptr)) 
				Notify(dbrefUser(ptr->shipdata->tactical),
				  "Lock achieved.");
			if (EventDriven(ptr))
				evLockAchieved(ptr);
		}

		if (Ship(ptr)) {

			if (ptr->shipdata->cloak_status == CLOAKING) {
				if (--ptr->shipdata->time_to_cloak <= 0) {
					ptr->tagflags |= CLOAKED;
					ptr->shipdata->cloak_status = CLOAK_ON;
					Notify(dbrefUser(ptr->shipdata->nav),
					  "Cloaking device engaged.");
				}
			}
			else if (ptr->shipdata->cloak_status == DECLOAKING) {
				if (--ptr->shipdata->time_to_cloak <= 0) {
					ptr->shipdata->cloak_status = CLOAK_OFF;
					Notify(dbrefUser(ptr->shipdata->nav),
					  "Cloaking device disengaged.");
				}
			}
	
		}

	}

	return;

}

void handle_tractor_target(TAG *source, TAG *target)
{
	int eff_tractor_power;
	float new_warp_speed;

	eff_tractor_power = target->shipdata->alloc_nav -
	  source->shipdata->tractor_power + distance(target->pos, source->pos);

	if (eff_tractor_power > 0) 
		new_warp_speed = navMaxWarp(target, eff_tractor_power);
	else
		new_warp_speed = 0.0;

	if (new_warp_speed < target->shipdata->warp_speed) {
		target->shipdata->warp_speed = new_warp_speed;
		FNotify(dbrefUser(target->shipdata->nav),
		  "Maximum safe warp speed is %3.2f.  Warp setting reduced.", 
		  new_warp_speed);
	}

	return;
			
}

void handle_tractor_source(TAG *source, TAG *target)
{

	int eff_tractor_power, move_power;
	float new_warp_speed;
	XYZ xyz;
	SPH sph;
	float dx, dy, dz;
	float dist, sum;

	eff_tractor_power = -target->shipdata->alloc_nav + 
	  source->shipdata->tractor_power - distance(source->pos, target->pos);

	move_power = Min(eff_tractor_power, source->shipdata->alloc_nav);

	if (move_power > 0)
		new_warp_speed = navMaxWarp(source, move_power);
	else
		new_warp_speed = 0.0;
				
	if (new_warp_speed < source->shipdata->warp_speed) {
		source->shipdata->warp_speed = new_warp_speed;
		FNotify(dbrefUser(source->shipdata->nav),
		  "Maximum safe warp speed is %3.2f.  Warp setting reduced.", 
		  new_warp_speed);
	}

	source->speed = source->shipdata->warp_speed;

	if (source->speed >= 1.0) {

		sph.bearing = source->heading.bearing;
		sph.elevation = source->heading.elevation;
		sph.range = source->speed * source->speed * source->speed;

		sph_to_xyz(sph, &xyz);
	
		target->pos.x += xyz.x;
		target->pos.y += xyz.y;
		target->pos.z += xyz.z;

	}
	
	if (source->shipdata->tractor_status == TRACTOR_DRAW) {

		move_power = eff_tractor_power - navWarpCost(source,
		  source->speed);

		if (move_power > 0) {
			new_warp_speed = navMaxWarp(target, move_power);

			dist = new_warp_speed * new_warp_speed * new_warp_speed;

			if (dist > distance(source->pos, target->pos)) {
				target->pos.x = source->pos.x;
				target->pos.y = source->pos.y;
				target->pos.z = source->pos.z;
			}
			else {
				dx = source->pos.x - target->pos.x;
				dy = source->pos.y - target->pos.y;
				dz = source->pos.z - target->pos.z;

				sum = sqrt(dx * dx + dy * dy  + dz * dz);
		
				target->pos.x += dx / sum * dist;
				target->pos.y += dy / sum * dist;
				target->pos.z += dz / sum * dist;

			}

		}

	}

	return;
}
