/* v0.9
 *
 * events.c:  Events code.
 *
 * This program is free software and may be freely redistributed as
 * specified in the GNU General Public License.  Please see the file
 * 'COPYING' for details.
 */

#include <stdlib.h>
#include <stdarg.h>
#include "spaceconf.h"
#include "pseint.h"
#include "dbint.h"
#include "space.h"
#include "object.h"
#include "events.h"
#include "sensors.h"
#include "smisc.h"

char *write_contact(CONTACT *);

char *write_contact(CONTACT *contact)
{
     char *arg;

     arg=AllocBuffer("write_contact");

     if (contact != NULL)
	  sprintf(arg, "%d", contact->contact_number);
     else
	  sprintf(arg, "-1");

     return arg;
}
	
void evCheckEvents(int space)
{
    
    range_t d;
    TAG *object;
    CONTACT *contact;
    
    for (object=space_info[space].list; object != NULL; object=object->next) {
	
	if (Removed(object))
	    continue;
	
	if (!EventDriven(object))
	    continue;
	
	for (contact = object->contact_list; contact != NULL; 
	     contact=contact->next) {
	    
	    /*
	     * Check for objects moving inside and outside
	     * critical range
	     */
	    
	    d = distance(&object->pos, &contact->listref->pos);
	    
	    if (contact->inside_critical && 
		d > object->critical_range) {
		contact->inside_critical = FALSE;
		evTrigger(object, EVENT_OUTSIDE_CRITICAL, "i",
			  contact->contact_number);
		if (space_info[object->space].flags & SPACE_LOGGED)
		    log_space("%s (#%d) is out of %s (#%d)'s critical "
			      "range at range " RANGEF ".", 
			      contact->listref->name, 
			      contact->listref->data_object, 
			      object->name, object->data_object, d);
	    } 
	    else if (!contact->inside_critical && 
		     d <= object->critical_range) {
		
		contact->inside_critical = TRUE;
		evTrigger(object, EVENT_INSIDE_CRITICAL, "i",
			  contact->contact_number);
		
		if (space_info[object->space].flags & SPACE_LOGGED)
		    log_space("%s (#%d) is in %s (#%d)'s critical range "
			      "at range " RANGEF ".", contact->listref->name, 
			      contact->listref->data_object,
			      object->name, object->data_object, d);
	    }
	    
	}
	
    }
    
    return;
    
}

void evListContainers(TAG *object, char *buff)
{
     char *ind = buff;
     TAG *ptr;
     CONTACT *con;

     strcpy(buff, "");

     for (ptr = space_info[object->space].list; ptr != NULL; ptr=ptr->next) {
	  for (con = ptr->contact_list; con != NULL; con=con->next) {

	       if (con->listref == object && con->inside_critical) {
		    sprintf(ind, "#%d ", ptr->data_object);
		    ind += strlen(ind);
	       }

	       /* Overflow prevention */
	       if (ind - buff >= LARGE_BUF_SIZE - 10)
		    return;
	  }
     }

     if (buff[strlen(buff) - 1] == ' ')
	  buff[strlen(buff) - 1] = '\0';

     return;
}

/*
 * See comments in message.c function msgSend for how to specify
 * the argument format.
 */
void evTrigger(TAG *object, const char *event, const char *argfmt, ...)
{
     va_list ap;
     unsigned int nargs, i;
     char **args;
     CONTACT *con; 
     TAG *obj = NULL;

     nargs = strlen(argfmt);	/* Determine number of arguments */
     args = (char **) pse_calloc(nargs, sizeof(char *));

     for (i = 0; i < nargs; i++) 
	  args[i] = (char *) pse_malloc(SMALL_BUF_SIZE);

     va_start(ap, argfmt);
     for (i = 0; i < nargs; i++) {
	  
	  switch (argfmt[i]) {
	  case 's':		/* String */
	       strncpy(args[i], va_arg(ap, char *), SMALL_BUF_SIZE);
	       args[i][SMALL_BUF_SIZE - 1] = '\0';
	       break;
	  case 'i':		/* Int */
	       sprintf(args[i], "%d", va_arg(ap, int));
	       break;
	  case 'f':		/* Float */
	       sprintf(args[i], "%f", va_arg(ap, double));
	       break;
	  case 'd':		/* Dbref */
	       obj = va_arg(ap, TAG *);
	       sprintf(args[i], "#%d", obj->data_object);
	       break;
          case 'c':		/* Contact */
	       con = va_arg(ap, CONTACT *);
	       if (con == NULL)
		    strcpy(args[i], "-1");
	       else
		    sprintf(args[i], "%d", con->contact_number);
	       break;

	  default:
	       /* Raise exception?  For now just don't send the message. */
	       return;
	       break;
	  }

     }
     va_end(ap);

     getEvalAttr(object->data_object, event, args, nargs);

     for (i = 0; i < nargs; i++) 
	  pse_free(args[i]);

     pse_free(args);

     return;

}

void evBroadcast(TAG *source, TAG *object, const char *msg,
		 const char *event)
{
     char *fargs[10];
     TAG *ptr;
     CONTACT *sourcecontact, *objcontact;
     CONTACT *contact;
     char buff[SMALL_BUF_SIZE], buff2[SMALL_BUF_SIZE];

     for (ptr = space_info[source->space].list; ptr != NULL; ptr=ptr->next) {

	  if (ptr == source || ptr == object) continue;

	  sourcecontact = objcontact = NULL;

	  for (contact = ptr->contact_list; contact != NULL; 
	       contact=contact->next) {
	       
	       if (contact->listref == source) 
		    sourcecontact = contact;

	       if (contact->listref == object)
		    objcontact = contact;

	  }

	  if (sourcecontact == NULL && objcontact == NULL)
	       continue;
	  
	  if (Ship(ptr) && msg != NULL) {

	       if (sourcecontact != NULL) 
		    sprintf(buff, "Contact [%d]:  %s",
			    sourcecontact->contact_number, source->name);

	       else
		    sprintf(buff, "An unknown source");
			
	       if (object != NULL) {

		    if (objcontact != NULL) {
			 sprintf(buff2, " %s [%d]",
				 object->name, 
				 objcontact->contact_number);
		    }
		    else
			 sprintf(buff2, " an unknown target");
	       }
	       else
		    strcpy(buff2, "");

	       MFNotify(ptr->shipdata, -1, CONS_TAC, CONS_ACTIVE,
		       "%s %s%s.", buff, msg, buff2); 

	  }	   
 
	  if (EventDriven(ptr) && event != NULL) {

	       fargs[0]=write_contact(sourcecontact);

	       if (object != NULL) 
		    fargs[1]=write_contact(objcontact);
	       else {
		    fargs[1]=AllocBuffer("evBroadcast");
		    sprintf(fargs[1], "-1");
	       }

	       getEvalAttr(ptr->data_object, event, fargs, 2);
	       FreeBuffer(fargs[0]);
	       FreeBuffer(fargs[1]);

	  }
     }

     return;
}

void evExplosion(int space, XYZ location, int damage)
{

     TAG *ptr;
     XYZ rel_pos;
     SPH bearing;
     float factor;

     factor = (float) damage / 500.0;

     for (ptr = space_info[space].list; ptr != NULL; ptr=ptr->next) {

	  if (CanSense(ptr) && distance(&location, &ptr->pos) < factor * 
	      (float) ptr->sensor_range) {

	       rel_pos.x = location.x - ptr->pos.x;
	       rel_pos.y = location.y - ptr->pos.y;
	       rel_pos.z = location.z - ptr->pos.z;
	       xyz_to_sph(rel_pos, &bearing);

	       if (Ship(ptr)) {
		    MFNotify(ptr->shipdata, -1, CONS_TAC, CONS_ACTIVE,
			    "An explosion was detected at "
			    "%3.1f%+3.1f, range %d.", bearing.bearing,
			    bearing.elevation, bearing.range);
	       }

	       if (EventDriven(ptr))
		    evTrigger(ptr, EVENT_EXPLOSION, "ii", damage, 
			      bearing.range);
	  }
     }
			
     return;
}

