/* v0.9
 *
 * object.c:  Space object code.
 *
 * This program is free software and may be freely redistributed as
 * specified in the GNU General Public License.  Please see the file
 * 'COPYING' for details.
 */

#include <stdlib.h>
#include <ctype.h>

#include "hashtab.h"
#include "spaceconf.h"
#include "pseint.h"
#include "dbint.h"
#include "space.h"
#include "object.h"
#include "damage.h"
#include "smisc.h"
#include "shields.h"
#include "events.h"
#include "nav.h"
#include "sensors.h"
#include "flag.h"

#ifdef ENABLE_SHIELD_CLASSES
#include "scm/scm.h"
#endif

static SHIP *load_ship(dbref);

void objInitSpace(void)
{
     return;
}

void objAddObject(dbref data)
{
    int space;
    TAG *new_object, *ptr;
    int start_dbref;
    
    if (objFindObject(data)) {
	log_space("AddObject Failure:  #%d:  Object already in space.", data);
	return;
    }
    
    new_object = (TAG *)pse_malloc(sizeof(TAG));
    
    /* Initialize the new object */
    
    new_object->data_object = data;
    space = new_object->space = atoi(getAttrByName(data, STATUS_SPACE));
    
    strncpy(new_object->name, strip_ansi(getObjectName(data)), NAME_LEN-1);
    new_object->name[NAME_LEN-1] = '\0';
    
    if (space < 0 || space >= NUM_SPACES) {
	log_space("AddObject Failure:  #%d:  Bad space number.  Valid "
		  "range is 0 to %d.", data, NUM_SPACES - 1);
	pse_free(new_object);
	return;
    }
    
    new_object->tagflags = 0;
    
    if (loadSymbolicFlags(&(new_object->tagflags), FLIST_OBJECT,
			  getAttrByName(data, STATUS_FLAGS)) != NULL) {
	
	log_space("#%d: Non-symbolic object flags found. Trying old-style.",
		  data);
	if (loadBinaryFlags(&(new_object->tagflags),
			    getAttrByName(data, STATUS_FLAGS)) == 0) {
	    log_space("#%d: Not binary flags either. Giving up.", data);
	    pse_free(new_object);
	    return;
	}
	
	log_space("#%d: Non-symbolic object flags updated [%s].",
		  data, getAttrByName(data, STATUS_FLAGS));
	setAttrByName(data, STATUS_FLAGS,
		      saveSymbolicFlags(new_object->tagflags, FLIST_OBJECT,
					0));
    }
    
    start_dbref = parseDbref(getAttrByName(data, STATUS_DOCKED_AT));
    
    ptr = NULL;
    if (start_dbref >= 0) 
	ptr = objFindObject(start_dbref);
    
    if (ptr != NULL) {
	new_object->pos.x = ptr->pos.x;
	new_object->pos.y = ptr->pos.y;
	new_object->pos.z = ptr->pos.z;
    }
    else
	fillXYZCoords(data, STATUS_STARTPOS, &(new_object->pos));
    
    new_object->range = HubRange(new_object->pos);
    
    new_object->heading.bearing = 0.0;
    new_object->heading.elevation = 0.0;
    new_object->heading.range = 1;
    
    new_object->heading_adj.bearing = 0;
    new_object->heading_adj.elevation = 0;
    new_object->heading_adj.range = 0;
    
    new_object->v_move[0] = 1.0;
    new_object->v_move[1] = 0.0;
    new_object->v_move[2] = 0.0;
    
    new_object->speed = 0.0;
    new_object->size = atof(getAttrByName(data, CONFIG_SIZE));
    new_object->tractor_source = NULL;
    
    new_object->moveID = move_turn_id;
    new_object->range_prev = NULL;
    new_object->range_next = NULL;
    new_object->huge_prev = NULL;
    new_object->huge_next = NULL;
    
    new_object->roll = 0.0;
    new_object->roll_adj = 0.0;
    
    if (new_object->size <= 0) {
	log_space("AddObject Failure:  #%d:  Object size must be "
		  "greater than zero.", data);
	pse_free(new_object);
	return;
    }
    
    new_object->sensor_range = atoi(getAttrByName(data, 
						  CONFIG_SENSOR_RANGE));
    new_object->transporter_range = atoi(getAttrByName(data, 
						       CONFIG_XPORT_RANGE));
    new_object->critical_range = atoi(getAttrByName(data, 
						    CONFIG_CRITICAL_RANGE));
    new_object->cloak_effect = atof(getAttrByName(data, 
						  CONFIG_CLOAK_EFFECT));
    
    new_object->pending_lock = NULL;
    new_object->locked_on = NULL;
    new_object->contact_list = NULL;
    
    getAttrByNameLen(new_object->range_name, data, CONFIG_RANGE_NAME, \
		     NAME_LEN-1);
    new_object->range_name[NAME_LEN-1] = '\0';
    new_object->range_factor = atof(getAttrByName(data, CONFIG_RANGE_FACTOR));
    if (new_object->range_factor == 0.0)
	new_object->range_factor = 1;
    
    if (Ship(new_object)) {
	if ((new_object->shipdata = load_ship(data)) == NULL) {
	    pse_free(new_object);
	    return;
	}
	
	damInitSystemDamage(new_object);
    }
    else
	new_object->shipdata = NULL;
    
    /* Make sure sensor range is nonzero */
    if (new_object->sensor_range == 0.0) {
	log_space("AddObject Failure:  #%d:  Bad sensor range", data);
	return;
    }
    
    /* Link in new object */
    snsAddObject(new_object, space);
    
    if (space_info[space].flags & SPACE_LOGGED)
	log_space("Added %s (#%d) to real space.", 
		  new_object->name,
		  data); 
    
    evTrigger(new_object, EVENT_STARTUP, "i", space);
    
    return;   
}

void objRemoveObject(TAG *object)
{

     evTrigger(object, EVENT_SHUTDOWN, "");

     object->tagflags |= REMOVED;

     return;

}	

void objUpdateState(TAG *object, dbref player)
{
    char buff[MAX_ATTRIBUTE_LEN];
    
    if (!ValidObject(object->data_object)) {
	
	FNotify(player, "#%d: Data object was not found. Data not updated.",
		object->data_object);
	return;
	
    }
    
    if (Ship(object)) {
	
	damWriteDamage(object->data_object, object->shipdata);
	
#ifdef ENABLE_ODOMETER
	navWriteOdometer(object->data_object, object->shipdata);
#endif
	
#ifdef ENABLE_TIME_TRACKER
	navWriteTimeActive(object->data_object, object->shipdata);
#endif
	
    }
    
    sprintf(buff, RANGEF " " RANGEF " " RANGEF,
	    object->pos.x, object->pos.y, object->pos.z);
    setAttrByName(object->data_object, STATUS_STARTPOS, buff);

    log_space("#%d: Wrote current attributes.", object->data_object);
    FNotify(player, "#%d: Wrote current attributes.", object->data_object);
}

void objFreeMarkedObjects(int space)
{
    TAG *object, *prev;
    TAG *ptr;
    
    object = space_info[space].list;
    prev = NULL;
    
    while(object != NULL) {
	/* If we are removing this object */
	if (Removed(object)) {
	    
	    /* Delete any pending timers */
	    del_timer(object, NULL);
	    
	    /* Remove ships from contact lists of other ships in
	       its space */
	    for (ptr = space_info[space].list; ptr != NULL; ptr=ptr->next)
		if (ptr != object)
		    snsRemoveContact(ptr, object, 0);
	    
	    /* Remove ship's own contacts */
	    while (object->contact_list != NULL)
		snsRemoveContact(object, object->contact_list->listref, 1);
	    
	    /* remove object from space list */
	    if (prev)
		prev->next = object->next;
	    else
		space_info[space].list = object->next;
	    
	    if (Ship(object)) {
		damWriteDamage(object->data_object, object->shipdata);
		
#ifdef ENABLE_ODOMETER
		navWriteOdometer(object->data_object, object->shipdata);
#endif
		
#ifdef ENABLE_TIME_TRACKER
		navWriteTimeActive(object->data_object, object->shipdata);
#endif
		
		pse_free(object->shipdata);
	    }
	    
	    if (space_info[space].flags & SPACE_LOGGED)
		log_space("Removed %s (#%d) from real space.", 
			  object->name, object->data_object); 
	    
	    ptr = object->next;
	    
	    snsDelObject(object, space);
	    pse_free(object);
	    
	    object = ptr;
	}
	else {
	    prev = object;
	    object = object->next;
	}
    }
    
    return;
}

SHIP *load_ship(dbref data)
{
    char buff[MAX_ATTRIBUTE_LEN];
    SHIP *ship;
    int i, t1, t2, invalid, consoles;
    const char *cn;
    const char *ct;
    dbref consoledb;
    
    ship = pse_malloc(sizeof(SHIP));
    
    ship->turn_factor = atof(getAttrByName(data, CONFIG_TURN_FACTOR));
    if (ship->turn_factor <= 0.0) {
	
	log_space("#%d: Invalid turn factor (%3.2f). Must be greater than 0."
		  " Default (1.0) used.", data, ship->turn_factor);
	ship->turn_factor = 1.0;
    }
    
    ship->roll_factor = atof(getAttrByName(data, CONFIG_ROLL_FACTOR));
    
    if (ship->roll_factor <= 0.0) {
	log_space("#%d: Invalid roll factor (%3.2f). Must be greater than "
		  "0. Default (1.0) used.", data, ship->roll_factor);
	ship->roll_factor = 1.0;
    }
    
    /* Load the shield configuration used by this ship */
    
    ship->shield_config = atoi(getAttrByName(data, CONFIG_SHIELD_CONFIG));
    if ((ship->shield_config < 0) ||
	(ship->shield_config >= NUM_SHIELD_CONFIGS)) {

	log_space("#%d:  Invalid shield config. Must be between 0 and %d.", 
		  data, NUM_SHIELD_CONFIGS-1);
	pse_free(ship);
	return NULL;
	
    }
    
    switch(ship->shield_config) {
    case SHIELD_CONFIG_ORIG4:
    case SHIELD_CONFIG_ORANGE1:
    case SHIELD_CONFIG_ORANGE2:
	ship->shield_number = 4;
	break;
	
    case SHIELD_CONFIG_ORIG6:
    case SHIELD_CONFIG_CUBE:
	ship->shield_number = 6;
	break;
    }
    
#ifdef ENABLE_ODOMETER
    ship->odometer = atoi(getAttrByName(data, STATUS_ODOMETER));
#endif
    
#ifdef ENABLE_TIME_TRACKER
    ship->time_active = atoi(getAttrByName(data, STATUS_TIME_ACTIVE));
    ship->time_start = time(NULL);
#endif
    
    /* Load the bridge and engineering emit locations */
    
    ship->dbref_bridge = getDbrefByName(data, CONFIG_DBREF_BRIDGE);
    ship->dbref_eng = getDbrefByName(data, CONFIG_DBREF_ENG);
    
    if (!ValidObject(ship->dbref_bridge)) {
	
	log_space("#%d: Invalid bridge dbref (#%d). Object not loaded.",
		  data, ship->dbref_bridge);
	pse_free(ship);
	return NULL;

    }

    if (!ValidObject(ship->dbref_eng)) {
	
	log_space("#%d: Invalid engineering dbref (#%d). Object not loaded.",
		  data, ship->dbref_eng);
	pse_free(ship);
	return NULL;

    }

    /* Load the consoles */
    
    getAttrByNameLen(buff, data, CONFIG_CONSOLE_LIST, MAX_ATTRIBUTE_LEN);
    
    cn = buff;
    consoles = invalid = 0;
    
    while((NULL != (cn = next_dbref(cn, &consoledb))) &&
	  (consoles < MAX_CONSOLES)) {
	
	if (!ValidObject(consoledb)) {

	    log_space("#%d: Invalid console dbref (Console #%d).", 
		      data, ship->consoles[consoles].dbref);

	    invalid = 1;

	} else {
	    
	    ship->consoles[consoles].dbref = consoledb;
	    
	    ct = getAttrByName(consoledb, CONFIG_CONSOLE_TYPES);
	    
	    if ((*ct == '\0') ||
		(ct = loadSymbolicFlags(&ship->consoles[consoles].flags,
					FLIST_CONSOLE, ct)) != NULL) {
		
		log_space("#%d: Invalid console type (#%d:%s).", 
			  data, ship->consoles[consoles].dbref, ct);
		
		invalid = 1;
		
	    } else
		consoles ++;
	}
    }
    
    if (consoles == 0) {

	log_space("#%d: No (valid) consoles were present. Object not loaded.", 
		  data);
	pse_free(ship);
	return NULL;
	
    }
    
    if (invalid) {
	
	log_space("#%d: Invalid console(s) found. Object not loaded.",  data);
	pse_free(ship);
	return NULL;
    }
    
    ship->num_consoles = consoles;

    /* initialize damage registers */
    
    getAttrByNameLen(buff, data, STATUS_SYSTEM_DAMAGE, MAX_ATTRIBUTE_LEN);
    if (strlen(buff) != NUM_SYSTEMS) {
	
	log_space("#%d: System damage string invalid. "
		  "Actual/Expected Length %d/%d.",
		  data, strlen(buff), NUM_SYSTEMS);
	pse_free(ship);
	return NULL;
	
    }
    
    for (i = 0; i < NUM_SYSTEMS; i++) {
	if (buff[i] == 'X')
	    buff[i] = 'x';
	
	if (((buff[i] < '0') || (buff[i] > '9'))
	    && (buff[i] != 'x'))
	    ship->damage[i].status = '0';
	else
	    ship->damage[i].status = buff[i];
	
	ship->damage[i].teams = 0;
	ship->damage[i].maxteams = 2;
	ship->damage[i].time_to_fix = 0;
	
    }
    
    ship->damage[SYSTEM_HULL].maxteams = 4;
    
    /* Load ship flags */

    ship->shipflags = 0;

    getAttrByNameLen(buff, data, STATUS_SHIPFLAGS, MAX_ATTRIBUTE_LEN);
    
    if (*buff != '\0') {
	
	ct = loadSymbolicFlags(&(ship->shipflags), FLIST_SHIP, buff);
	
	if (ct != NULL) {
	    
	    log_space("#%d: Non-symbolic ship flags found. Trying old-style.",
		      data);
	    
	    
	    if (loadBinaryFlags(&(ship->shipflags),
				getAttrByName(data, STATUS_SHIPFLAGS)) == 0) {
		
		log_space("#%d: Not binary flags either. Giving up.", data);
		pse_free(ship);
		return NULL;
	    }
	    
	    /* Write out symbolic flags */
	    
	    log_space("#%d: Non-symbolic ship flags replaced. [%s].",
		      data, getAttrByName(data, STATUS_SHIPFLAGS));
	    setAttrByName(data, STATUS_SHIPFLAGS,
			  saveSymbolicFlags(ship->shipflags, FLIST_SHIP, 0));
	}
    }
        
    getAttrByNameLen(ship->class, data, CONFIG_CLASS, CLASS_LEN);
    getAttrByNameLen(ship->type, data, CONFIG_TYPE, TYPE_LEN-1);
    getAttrByNameLen(ship->owner_name, data, CONFIG_OWNER_NAME, OWNER_LEN);
    ship->owner = atoi(getAttrByName(data, CONFIG_OWNER_NUM));
    
    /* Docking bay doors */
    
    ship->door_status = atoi(getAttrByName(data, CONFIG_DOOR_STATUS))
	? DOORS_CLOSED : DOORS_NONE; 
    
    if (ship->door_status != DOORS_NONE) {
	
	ship->door_size = atof(getAttrByName(data, CONFIG_DOOR_SIZE));
	ship->door_side = atoi(getAttrByName(data, CONFIG_DOOR_SIDE));	
	
	if (ship->door_side < 0 || ship->door_side >= NUM_SHIELDS) {
	    log_space("#%d: Door side must be between 0 and %d.",
		      data, NUM_SHIELDS-1);
	    pse_free(ship);
	    return NULL;
	}
    }
    
    /* Engine stuff */
    
    ship->hull_integrity = atoi(getAttrByName(data, CONFIG_HULL));
    ship->reactor_output_max =
	atoi(getAttrByName(data, CONFIG_REACTOR_OUTPUT));
    
    ship->max_overload_points =
	atoi(getAttrByName(data, CONFIG_REACTOR_OVER_POINTS));
    
    ship->reactor_setting_limit =
	atoi(getAttrByName(data, CONFIG_REACTOR_LEVEL_STRESS));
    
    ship->reactor_stress_level =
	atoi(getAttrByName(data, CONFIG_REACTOR_LEVEL_NORMAL));

    ship->reactor_overload_penalty =
	atoi(getAttrByName(data, CONFIG_REACTOR_OVER_PENALTY));
    
    /* battery stuff */
    
    ship->battery_capacity =
	atoi(getAttrByName(data, CONFIG_BATTS_CAPACITY));
    
    ship->battery_discharge_max =
	atoi(getAttrByName(data, CONFIG_BATTS_DISCHARGE));
    
    /* general ship characteristics */

    ship->warp_factor = atof(getAttrByName(data, CONFIG_WARP_FACTOR));
    ship->warp_rated_max = atof(getAttrByName(data, CONFIG_WARP_MAX));
    ship->warp_max = 0;
    ship->warp_accel = atof(getAttrByName(data, CONFIG_WARP_ACCEL));
    if (ship->warp_accel == 0.0)
	log_space("#%d: Warp acceleration is 0.0. Ship cannot move!", data);
    
    /* cloak */
    ship->cloak_status = atoi(getAttrByName(data, CONFIG_CLOAK_STATUS)) 
	? CLOAK_OFF : CLOAK_NONE;
    ship->time_to_cloak = 0;
    ship->cloak_cost = atoi(getAttrByName(data, CONFIG_CLOAK_COST));
    
    /* sensors */
    ship->scanner_range = atoi(getAttrByName(data, CONFIG_SCANNER_RANGE));
    
    /* comm */
    ship->transmitter_range = atof(getAttrByName(data, CONFIG_COMM_RANGE));
    ship->receiver_sensitivity = atof(getAttrByName(data, 
						    CONFIG_COMM_SENSITIVITY));
    
#ifndef ENABLE_SHIELD_CLASSES
    /* shield stuff */
    ship->shield_factor = atof(getAttrByName(data, CONFIG_SHIELD_FACTOR));
    
    for (i = 0; i < NUM_SHIELDS; i++) {
	ship->shield_max[i] =
	    atoi(getAttrByName(data, CONFIG_SHIELD_POWER));
	
	ship->shield_max_charge[i] =
	    atoi(getAttrByName(data, CONFIG_SHIELD_CHARGE_RATE));
    }
#else
    /* shield class initialization */
    
    /* Clean out the shield settings */
    for (i = 0; i < SCM_MAX_LAYERS; i ++) {
	ship->sc[i].name[0] = '\0';
	ship->sc[i].properties = NULL;
	ship->sc[i].storage = NULL;
	ship->sc[i].geometry.shields = 0;
	ship->sc[i].attrib = -1;
    }
    
    ship->sc_num = 0;
    
    /* Attempt to load in the next shield layer */
    for (i = 0; i < SCM_MAX_LAYERS; i ++) {
	
	sprintf(buff, "%s%d", SHIELDARRAY_LAYER, i);
	
        getAttrByNameLen(buff, data, buff, MAX_ATTRIBUTE_LEN);
	
	if (scmLoadLayer(data, &ship->sc[ship->sc_num], buff)) {

	    ship->sc[ship->sc_num].attrib = i;
	    ship->sc_num ++;

	}
    }
#endif
    
    /* gun stuff */

    getAttrByNameLen(ship->gun_string, data, CONFIG_GUN_NAME, GUN_LEN);

    ship->number_of_guns = atoi(getAttrByName(data, CONFIG_GUN_QTY));
   
    if (ship->number_of_guns > MAX_GUNS) {
	
	log_space("#%d: Too many guns specified. Using %d.", data, MAX_GUNS);
	
	ship->number_of_guns = MAX_GUNS;
    }
    
    for (i = 0; i < ship->number_of_guns; i++) {
	
	ship->gun[i].status = GUN_OFFLINE;
	ship->gun[i].power = atoi(getAttrByName(data, CONFIG_GUN_POWER));
	ship->gun[i].range = atoi(getAttrByName(data, CONFIG_GUN_RANGE));
	
	ship->gun[i].range_percent =
	    atof(getAttrByName(data, CONFIG_GUN_DELIVERY));
	
	ship->gun[i].charge_per_turn = 
	    atoi(getAttrByName(data, CONFIG_GUN_CHARGE_RATE));

	ship->gun[i].arc = atoi(getAttrByName(data, CONFIG_GUN_ARC));

    }
    
    /* torp stuff */

    getAttrByNameLen(ship->torp_string, data, CONFIG_TORP_NAME, TORP_LEN);

    ship->number_of_torps = atoi(getAttrByName(data, CONFIG_TORP_QTY));

    if (ship->number_of_torps > MAX_TORPS) {

	log_space("#%d: Too many torps specified. Using %d.", data, MAX_TORPS);

	ship->number_of_torps = MAX_TORPS;
    }
    
    getAttrByNameLen(buff, data, CONFIG_TORP_AMMO, MAX_ATTRIBUTE_LEN);
    if (*buff != '\0')
	ship->torp_ammo = atoi(buff);
    else
	ship->torp_ammo = -1;
    
    for (i = 0; i < ship->number_of_torps; i++) {
	
	ship->torp[i].power = atoi(getAttrByName(data, CONFIG_TORP_POWER));
	ship->torp[i].range = atoi(getAttrByName(data, CONFIG_TORP_RANGE));

	ship->torp[i].charge_per_turn =
	    atoi(getAttrByName(data, CONFIG_TORP_CHARGE_RATE));
	
	ship->torp[i].base_accuracy =
	    atof(getAttrByName(data, CONFIG_TORP_ACCURACY));
    }
    
    ship->num_torp_loaders = atoi(getAttrByName(data, CONFIG_TORP_LOADERS));
    
    ship->free_torp_loaders = ship->num_torp_loaders;
    ship->tractor_status = atoi(getAttrByName(data, CONFIG_TRACTOR_STATUS))
	? TRACTOR_OFF : TRACTOR_NONE; 
    
    if (ship->tractor_status != TRACTOR_NONE)
	ship->tractor_effect = atof(getAttrByName(data, 
						  CONFIG_TRACTOR_EFFECT));
    
    /* initialize damage control team status */
    ship->damcon_teams = atoi(getAttrByName(data, CONFIG_DAMCON_TEAMS));

    if (ship->damcon_teams > MAX_TEAMS) {

	log_space("#%d: Too many damage control teams specified. Using %d.",
		  data, MAX_TEAMS);
	
	ship->damcon_teams = MAX_TEAMS;
	
    }
    
    sscanf(getAttrByName(data, STATUS_HULL_DAMAGE), "%d %d", 
	   &t1, &t2);
    ship->current_integrity = ship->hull_integrity - t1;
    ship->permanent_hull_damage = t2;
    
    if (ship->current_integrity < 0)
	ship->shipflags |= DISABLED;
    
    ship->reactor_setting = 5;
    ship->reactor_drain = 0;
    ship->overload_points = 0;
    ship->eng_warnings = 1; /* set warnings on */
    ship->eng_safeties = 1; /* set automatic overrides on */
    ship->battery_discharge = 0;
    ship->battery_level = ship->battery_capacity;
    
    /* cloak */
    if (ship->cloak_status == CLOAK_NONE)
	ship->damage[SYSTEM_CLOAK].status = 'x';
    
    /* tractor */
    if (ship->tractor_status == TRACTOR_NONE)
	ship->damage[SYSTEM_TRACTOR].status = 'x';
    
    /* default allocations */
    ship->alloc_nav = 0;
    ship->alloc_batt = 0;
    ship->alloc_tac = 0;
    ship->alloc_nav = 0;
    ship->alloc_shields = 0;
    ship->talloc_guns = 0;
    ship->talloc_torps = 0;
    ship->talloc_tractor = 0;
    ship->tractor_target = NULL;
    ship->tractor_power = 0;
    ship->num_guns_online = 0;
    ship->num_torps_online = 0;
    
    ship->warp_speed = 0.0;
    ship->warp_set_speed = 0.0;
    
    /* shield stuff */
    for (i = 0; i < NUM_SHIELDS; i++) {
	ship->shield_level[i] = 0;
	ship->shield_alloc[i] = 0;
	ship->shield_status[i] = SHLD_READY;
	ship->shield_action[i] = SHLD_STABLE;
    }
    
    /* gun stuff */
    for (i = 0; i < ship->number_of_guns; i++) {
	ship->gun[i].status = GUN_OFFLINE;
	ship->gun[i].charge = 0;
    }
    
    for (i = ship->number_of_guns; i < MAX_GUNS; i++)
	ship->damage[SYSTEM_GUN_0 + i].status = 'x';
    
    /* torp stuff */
    ship->auto_online = FALSE;
    ship->auto_reload = TRUE;
    
    for (i = 0; i < ship->number_of_torps; i++) {
	ship->torp[i].status = TORP_LOADED;
	ship->torp[i].charge = 0;
	ship->torp[i].turns_charged = 0;
    }
    
    for (i = ship->number_of_torps; i < MAX_TORPS; i++)
	ship->damage[SYSTEM_TORP_0 + i].status = 'x';
    
    /* initialize comm configuration */
    for (i = 0; i < MAX_COMM_CHANNELS; i++) {
	ship->channel[i].frequency = 0;
	ship->channel[i].send_status = CS_COMM_ONLY;
	ship->channel[i].receive_status = CR_COMM_ONLY;
	strcpy(ship->channel[i].label, "");
    }
    
    /* initialize damage control team status */
    for (i = 0; i < ship->damcon_teams; i++)
	ship->team[i] = -1;
    
    if (ship->max_overload_points == 0)
	log_space("#%d:  Bad max_overload_points (0).", data);
    else if (ship->warp_factor == 0.0)
	log_space("#%d:  Bad warp_factor (0.0).", data);
    else if (ship->hull_integrity <= 0)
	log_space("#%d:  Bad hull_integrity (%d)", data, ship->hull_integrity);
    else
	return ship;  /* everything checks out */
    
    /* if it gets to here there is a problem with the ship object */
    pse_free(ship);
    return NULL;
    
}

void objDisplaySpecs(TAG *object, dbref player, int godmode)
{
     SHIP *ship = object->shipdata;

     if (Ship(object))
	  FNotify(player, "%s -- %s %s-class %s", object->name,
		  ship->owner_name, ship->class, ship->type);
     else
	  FNotify(player, "%s", object->name);

     if (godmode) {
	  FNotify(player, "Flags: %s",
		  saveSymbolicFlags(object->tagflags, FLIST_OBJECT, 0));

	  FNotify(player, "Size: %2.1f", object->size);
     }

     if (CanSense(object))
	  FNotify(player, "Sensor Range: %.f %s",
		  object->sensor_range * object->range_factor,
		  object->range_name);

     if (object->transporter_range > 0)	
	  FNotify(player, "Transporter Range: %d", object->transporter_range);

     if (EventDriven(object))
	  FNotify(player, "Critical Range: %.f",
		  object->critical_range * object->range_factor,
		  object->range_name);

     if (object->cloak_effect > 0)
	  FNotify(player, "Cloak Effectiveness: %1.1f", 
		  object->cloak_effect);

     if (Ship(object)) {

	  if (godmode)
	      FNotify(player, "Ship Flags: %s", 
		      saveSymbolicFlags(ship->shipflags, FLIST_SHIP, 0));
	  FNotify(player, "Engine output max: %d    Optimum running "
		  "level: %d%%", ship->reactor_output_max, 
		  ship->reactor_stress_level);
	  FNotify(player, "Battery capacity: %d    Discharge rate: %d", 
		  ship->battery_capacity, ship->battery_discharge_max);

	  FNotify(player, "Turn factor: %4.2f          Roll factor: %4.2f", 
		  ship->turn_factor, ship->roll_factor);

	  if (ship->door_status != DOORS_NONE)
	       FNotify(player, "Hull integrity: %5d      Door size: "
		       "%2.1f", ship->hull_integrity, ship->door_size);
	  else
	       FNotify(player, "Hull integrity: %d", ship->hull_integrity);
 
	  if (ship->cloak_status != CLOAK_NONE)
	       FNotify(player, "Cloaking Device: Yes       Cost: %d", 
		       ship->cloak_cost);
	  else
	       Notify(player, "Cloaking Device: No");

	  if (ship->tractor_status != TRACTOR_NONE)	
	       FNotify(player, "Tractor Beam: Yes          "
		       "Effectiveness: %3.1f", ship->tractor_effect);
	  else
	       Notify(player, "Tractor Beam: No");

	  FNotify(player, "Scanner Range: %.f %s",
		  ship->scanner_range * object->range_factor,
		  object->range_name);

	  FNotify(player, "Shield Array: %s", shdConfigName(object));

	  FNotify(player, "Maximum normal shield charge: %d", 
		  ship->shield_max[0]);

	  if (ship->number_of_guns)
	       FNotify(player, "%s banks: %d   Maximum charge: %d   "
		       "Effective range: %d",
		       capstr(ship->gun_string), ship->number_of_guns,
		       ship->gun[0].power,
		       (int)(0.8 * (float)(ship->gun[0].range)));
	  else
	       FNotify(player, "%s banks: 0", 
		       capstr(ship->gun_string));

	  if (ship->number_of_torps) 
	       FNotify(player, "%s tubes: %d   Arm: %d  "
		       "Power: %d  Effective range: %d  Reloaders: %d", 
		       capstr(ship->torp_string), ship->number_of_torps,
		       ship->torp[0].power / 2, ship->torp[0].power,
		       (int)(0.6 * (float)(ship->torp[0].range)),
		       ship->num_torp_loaders);
	  else
	       FNotify(player, "%s tubes: 0", 
		       capstr(ship->torp_string));

     }

     return;
}

/* Spacegod commands */
void objList(dbref player, int space, int show_ships, int show_other)
{
     int count;
     SPH bearing;
     TAG *ptr;

     Notify(player, space_info[space].name);
     Notify(player, "---------------------------------------------"
	    "-------------------------------");

     count = 0;

     for (ptr = space_info[space].list; ptr != NULL; ptr=ptr->next) {

	  count++;
	  if (!show_ships && Ship(ptr)) continue;
	  if (!show_other && !Ship(ptr)) continue;

	  xyz_to_sph(ptr->pos, &bearing);

	  if (!Ship(ptr) && ptr->speed==0) 
	       FNotify(player,"#%-5d %-30s  %3.0f%-+4.0f %10d\t\t%c%c", 
		       ptr->data_object, ptr->name,
		       bearing.bearing, bearing.elevation, bearing.range,
		       (ptr->locked_on != NULL) ? 'L' : ' ',
		       Cloaked(ptr) ? 'C' : ' ');
	  else
	       FNotify(player, "#%-5d %-30s  %3.0f%-+4.0f %10d  %3.0f"
		       "%-+4.0f %4.1f\t%c%c",ptr->data_object, ptr->name, 
		       bearing.bearing, bearing.elevation, 
		       bearing.range, ptr->heading.bearing, 
		       ptr->heading.elevation, ptr->speed,
		       (ptr->locked_on != NULL) ? 'L' : ' ',
		       Cloaked(ptr) ? 'C' : ' ');

     }

     Notify(player, "---------------------------------------------"
	    "-------------------------------");
     FNotify(player, "Total objects:  %d", count);
     Notify(player, " ");

     return;
}

TAG *objFindObject(dbref target)
{
     int i;
     TAG *object;

     for (i = 0; i < NUM_SPACES; i++) { 
	  for (object = space_info[i].list; object != NULL; 
	       object=object->next)
	       
	       if (object->data_object == target)
		    return(object);
     }

     return (NULL);

} 

/*
 * Matches the provided name with the system identification
 * strings.
 * 
 * Returns: -1 Ambiguous
 *	    -2 Unknown
 *	    0-NUM_SYSTEMS Matched System ID
 *
 */
int objSystemNameToInt(const TAG *object, char *system)
{
     SHIP *ship = object->shipdata;
     int found = 0;
     int sys;
     int i;
     unsigned int len;
     char str[MAX_ATTRIBUTE_LEN];

     len = strlen(system);

     if ((len == 0) || (!Ship(object)))
	  return -2;

     strncpy(str, system, MAX_ATTRIBUTE_LEN);
     str[MAX_ATTRIBUTE_LEN-1] = '\0';

     str[0] = toupper(str[0]);
     for (i = 1; str[i]; i++)
	  str[i] = tolower(str[i]);

     for (i=sys=0; i < ship->shield_number; i++)
	  if (strncmp(str, shdFullName(object,i), len) == 0) {
	       sys = i;
	       found++;
	  }

     str[0] = tolower(str[0]);

     /* figure out which system we're fixing */
     for (i=NUM_SHIELDS; i < NUM_SYSTEMS; i++) 
	  if (strncmp(str, system_names[i], len) == 0) {
	       sys = i;
	       found++;
	  }

     if (!found)
	  /* we didn't find the system.*/
	  return -2;

     if (found != 1)
	  /* Found more than 1 */
	  return -1;

     return sys;
}

const char* objSystemIntToName(const TAG *object, int system)
{
     if (!Ship(object))
	  return "";

     if (system < NUM_SHIELDS)
	  return shdFullName(object,system);

     if (system >= NUM_SYSTEMS)
	  return "";

     return system_names[system];
}

void objSetObjectFlag(TAG *object, dbref player, int value, char *name)
{
    if (setFlag(&(object->tagflags), value, FLIST_OBJECT, name))
	Notify(player, "Invalid flag.");
    else {
	if (strcmp(name, "SHIP") == 0) {
	    Notify(player, "#-1 SHIP FLAG CANNOT BE CHANGED");
	    return;
	}

	if (value)
	    Notify(player, "Set.");
	else
	    Notify(player, "Cleared.");
	if (space_info[object->space].flags & SPACE_LOGGED)
	    log_space("%s (#%d) set %s (#%d) %s%s.", getObjectName(player), 
		      player, object->name, object->data_object, 
		      value ? "" : "!", name);
    }
    
    return;
}

void objSetShipFlag(TAG *object, dbref player, int value, char *name)
{
    if (setFlag(&(object->shipdata->shipflags), value, FLIST_SHIP,
		name))
	Notify(player, "Invalid flag.");
    else {
	if (value)
	    Notify(player, "Set.");
	else
	    Notify(player, "Cleared.");
	
	if (space_info[object->space].flags & SPACE_LOGGED)
	    log_space("%s (#%d) set %s (#%d) %s%s.", getObjectName(player), 
		      player, object->name, object->data_object, 
		      value ? "" : "!", name);
    }
    
    return;
}

int objHasObjectFlag(const TAG *object, dbref player, const char *name)
{
    return (isFlagSet(object->tagflags, FLIST_OBJECT,  name));
}

int objHasShipFlag(const TAG *object, dbref player, const char *name)
{
    return (isFlagSet(object->shipdata->shipflags, FLIST_SHIP, name));
}

int objHasConsoleFlag(const TAG *object, dbref player, int console,
		      const char *name)
{
    return (isFlagSet(object->shipdata->consoles[console].flags,
		       FLIST_CONSOLE, name));
}
