/* v0.9
 *
 * platform.c:  Platform-specific code.
 *
 * This program is free software and may be freely redistributed as
 * specified in the GNU General Public License.  Please see the file
 * 'COPYING' for details.
 */

#include <stdlib.h>
#include <stdarg.h>
#include <ctype.h>
#include "spaceconf.h"
#include "platform.h"
#include "pseint.h"
#include "space.h"

/* Currently for Penn, TinyMux and TinyMush, the default is GOD */
dbref debug_char=GOD;

#if defined(MUX) || defined(MUSH)

/* 
 * evalString:  Evaluate a string using the platform's 'exec' function.
 */
void evalString(char *buff, dbref obj, char *str, char **cargs, 
		int ncargs)
{
     char *tmpbuf;

#ifdef MUX
     char *bp;
	
     tmpbuf = alloc_lbuf("getEvalAttr");

     bp = tmpbuf;
     exec(tmpbuf, &bp, 0, obj, GOD, EV_EVAL | EV_FIGNORE | EV_TOP,
	  &str, cargs, ncargs);
     *bp='\0';
#else
     tmpbuf = exec(obj, GOD, EV_EVAL | EV_FIGNORE | EV_TOP,
		   str, cargs, ncargs);
#endif

     strncpy(buff, tmpbuf, MAX_ATTRIBUTE_LEN);
     buff[MAX_ATTRIBUTE_LEN - 1] = '\0';

     /*
      * We must free the buffer we allocated (MUX), or that was allocated
      * for us (MUSH).
      */
     free_lbuf(tmpbuf);

     return;
}

/*
 * getAttr:  Fetch an attribute string into the volatile buffer.
 */
const char *getAttrByNumber(dbref obj, int attrnum)
{
     dbref aowner;
     int aflags;
     char *atr_gotten;
     static char buff[MAX_ATTRIBUTE_LEN];
     
     switch(attrnum) {
     case PSE_DATA_DBREF:
	 attrnum = A_DATA_DBREF;
	 break;
	 
     case PSE_USER_DBREF:
	 attrnum = A_USER_DBREF;
	 break;
     }

     atr_gotten = atr_pget(obj, attrnum, &aowner, &aflags);

     strncpy(buff, atr_gotten, MAX_ATTRIBUTE_LEN);
     buff[MAX_ATTRIBUTE_LEN - 1] = '\0';
     free_lbuf(atr_gotten);

     return(buff);
}

/*
 * getAttr:  Fetch an attribute string into the volatile buffer.
 */
void getAttrByNumberLen(char *buff, dbref obj, int attrnum, int max)
{
     dbref aowner;
     int aflags;
     char *atr_gotten;
     
     switch(attrnum) {
     case PSE_DATA_DBREF:
	 attrnum = A_DATA_DBREF;
	 break;
	 
     case PSE_USER_DBREF:
	 attrnum = A_USER_DBREF;
	 break;
     }

     atr_gotten = atr_pget(obj, attrnum, &aowner, &aflags);

     strncpy(buff, atr_gotten, max);
     buff[max - 1] = '\0';
     free_lbuf(atr_gotten);
}

const char *getAttrByName(dbref obj, const char *attrname)
{
     ATTR *attr;

#ifdef MUX
     if (!(attr=atr_str(attrname)))
	  return("");
     else
	  return getAttrByNumber(obj, attr->number);
#endif

#ifdef MUSH
     /*
      * Mush modifies the provided buffer in some circumstances, so we have
      * to take a copy.
      */
     char buff[MAX_ATTRIBUTE_LEN];

     strncpy(buff, attrname, MAX_ATTRIBUTE_LEN);
     buff[MAX_ATTRIBUTE_LEN-1]='\0';

     if (!(attr=atr_str(buff)))
	  return("");
     else
	  return getAttrByNumber(obj, attr->number);     
#endif
}

void getAttrByNameLen(char *buff, dbref obj, const char *attrname, int max)
{
     ATTR *attr;

#ifdef MUX
     if (!(attr=atr_str(attrname))) {
	  buff[0] = '\0';
	  return;
     } else
	  return getAttrByNumberLen(buff, obj, attr->number, max);
#endif

#ifdef MUSH
     /*
      * Mush modifies the provided buffer in some circumstances, so we have
      * to take a copy.
      */
     static char abuff[MAX_ATTRIBUTE_LEN];

     strncpy(abuff, attrname, MAX_ATTRIBUTE_LEN);
     abuff[MAX_ATTRIBUTE_LEN-1]='\0';

     if (!(attr=atr_str(abuff))) {
	  buff[0] = '\0';
	  return;
     } else
	  return getAttrByNumberLen(buff, obj, attr->number, max);     
#endif
}

/*
 * getEvalAttr:  Fetch an attribute string, and evaluate it
 */ 
	
const char *getEvalAttr(dbref obj, const char *attrname, char **cargs,
			unsigned int ncargs)
{
     static char buff[MAX_ATTRIBUTE_LEN];
     static char buff2[MAX_ATTRIBUTE_LEN];
     ATTR *attr;

#ifdef MUSH
     /*
      * Mush modifies the provided buffer, so we have to take a copy.
      */
     strncpy(buff, attrname, MAX_ATTRIBUTE_LEN);
     buff[MAX_ATTRIBUTE_LEN-1]='\0';

     if ((attr=atr_str(buff)) == NULL)
#endif
#ifdef MUX
     if ((attr=atr_str(attrname)) == NULL)
#endif
	  buff[0] = '\0';
	      
     else {

	  getAttrByNumberLen(buff2, obj, attr->number, MAX_ATTRIBUTE_LEN);

	  if (*buff2)
	       evalString(buff, obj, buff2, cargs, ncargs);
	  else
	       buff[0] = '\0';

     }

     return buff;

}

/*
 * setAttr:  Record an attribute in the database. 
 */
void setAttrByNumber(dbref obj, int attrnum, const char *value)
{
     dbref aowner;
     int aflags;
     static char buff[MAX_ATTRIBUTE_LEN];

     if (ValidObject(obj)) {
	  strncpy(buff, value, MAX_ATTRIBUTE_LEN);
	  buff[MAX_ATTRIBUTE_LEN - 1] = '\0';
	  atr_pget_info(obj, attrnum, &aowner, &aflags);
	  atr_add(obj, attrnum, buff, aowner, aflags);
     }

     return;
}
		
void setAttrByName(dbref obj, const char *attrname, const char *value)
{
     ATTR *attr;
#ifdef MUSH
     /*
      * Mush modifies the provided buffer in some circumstances, so we have
      * to take a copy.
      */
     char buff[MAX_ATTRIBUTE_LEN];

     strncpy(buff, attrname, MAX_ATTRIBUTE_LEN);
     buff[MAX_ATTRIBUTE_LEN-1]='\0';


     if ((attr=atr_str(buff)) != NULL)
	  setAttrByNumber(obj, attr->number, value);
#endif
#ifdef MUX
     if ((attr=atr_str(attrname)) != NULL)
	  setAttrByNumber(obj, attr->number, value);
#endif

     return;
}

/* This excerpt was taken from the MUX server code. */

int getAttrNumber(const char *name)
{
     ATTR *ap;

#ifdef MUSH
     /*
      * Mush modifies the provided buffer in some circumstances, so we have
      * to take a copy.
      */
     char buff[MAX_ATTRIBUTE_LEN];

     strncpy(buff, name, MAX_ATTRIBUTE_LEN);
     buff[MAX_ATTRIBUTE_LEN-1]='\0';

     if (!(ap = atr_str(buff)))
	  return 0;
#endif

#ifdef MUX
     if (!(ap = atr_str(name)))
	  return 0;
#endif

     if (!(ap->number))
	  return -1;

     return ap->number;
}

#endif

#ifdef PENN

/*
 * getAttr:  Fetch an attribute string into the volatile buffer.
 */
const char *getAttrByNumber(dbref obj, int attrnum)
{
     switch(attrnum)
     {
	case PSE_DATA_DBREF:
		return getAttrByName(obj, "DATADBREF");

	case PSE_USER_DBREF:
		return getAttrByName(obj, "USERDBREF");
     }

     return "";
}

void getAttrByNumberLen(char *buff, dbref obj, int attrnum, int max)
{
     switch(attrnum)
     {
	case PSE_DATA_DBREF:
	    getAttrByNameLen(buff, obj, "DATADBREF", max);
	    return;

	case PSE_USER_DBREF:
	    getAttrByNameLen(buff, obj, "USERDBREF", max);
	    return;
     }

     buff[0] = '\0';
     return;
}

const char *getAttrByName(dbref obj, const char *attrname)
{
     static char buff[MAX_ATTRIBUTE_LEN];
     ATTR *attr;
     char *filter;

     if (!(attr=atr_get(obj, attrname)))
	  return("");

     filter = safe_uncompress(attr->value);
     strncpy(buff, filter, MAX_ATTRIBUTE_LEN);
     buff[MAX_ATTRIBUTE_LEN - 1] = '\0';
     pse_free(filter);

     return buff;     
}

void getAttrByNameLen(char *buff, dbref obj, const char *attrname, int max)
{
     ATTR *attr;
     char *filter;

     buff[0] = '\0';

     if (!(attr=atr_get(obj, attrname)))
	  return;

     filter = safe_uncompress(attr->value);
     strncpy(buff, filter, max);
     buff[max - 1] = '\0';
     pse_free(filter);

     return;     
}

const char *getEvalAttr(dbref obj, const char *attrname, char **cargs,
			unsigned int ncargs)
{
    static char buff[BUFFER_LEN];
    ATTR *attr;
    char *temp = buff;
    
    buff[0] = '\0';
    
    if ((attr=atr_get(obj, attrname)) != NULL)
      	do_userfn(buff, &temp, obj, attr, ncargs, cargs, GOD, GOD, GOD, NULL);
    
    buff[MAX_ATTRIBUTE_LEN-1] = '\0';

    return buff;
}

void setAttrByName(dbref obj, const char *attrname, const char *value)
{

     do_set_atr(obj, attrname, value, GOD, NOTHING);

}

int getAttrNumber(const char *attr)
{
	/* TODO: Fix this if/when we use varrays. We probably need to hook
                 a Penn call to return a unique hash of the attribute. */
	return 1;
}

#endif

/*
 * Platform independant function to remove ansi from strings. MUX has this
 * already, so we don't define it again here.
 */

#if defined(MUSH) || defined(PENN)
char *strip_ansi(const char *str)
{
    static char buff[MAX_ATTRIBUTE_LEN];
    const char *s = str;
    char *t = buff;
    
    while (s && *s) {
	if (*s == 0x1b) {
	    while (*s && !isalpha(*s))
		s++;
	    if (*s)
		s++;
	} else
	    *t++ = *s++;
    }
    *t = '\0';
    return buff;
}
#endif

/*
 * Platform indepenant interface functions to output responses.
 */

void Notify(dbref obj, const char *str)
{
    if (obj == PSE_NOTHING)
	return;
    
#if defined(MUX) || defined(MUSH)
    notify_with_cause(obj, GOD, str);
#endif
#ifdef PENN
    notify_by(GOD, obj, str);
#endif
}

void NotifyExcept(dbref loc, dbref except, const char *str)
{
#if defined(MUX) || defined(MUSH)
    if (except == PSE_NOTHING)
	except = NOTHING;

    notify_except(loc, GOD, except, str);
#endif

#ifdef PENN
    dbref ortemp;

    if (except == PSE_NOTHING)
	except = NOTHING;

    ortemp = orator;
    orator = GOD;
    notify_except(loc, except, str);
    orator = ortemp;
#endif
}

void NotifyLocation(dbref loc, const char *str)
{
#if defined(MUX) || defined(MUSH)
    if (loc == PSE_NOTHING)
	return;
    
    notify_except(loc, GOD, NOTHING, str);
#endif

#ifdef PENN
    dbref or = orator;

    if (loc == PSE_NOTHING)
	return;
    
    orator = GOD;
    notify_except(loc, NOTHING, str);
    orator = or;
#endif
}

/*
 * Platform independant functions used by the PSE to allocate and free buffers.
 */
void *AllocBuffer(const char *id)
{
#if defined(MUX) || defined(MUSH)
    return alloc_sbuf(id);
#endif

#ifdef PENN
    return pse_malloc(MAX_ATTRIBUTE_LEN);
#endif
}

void FreeBuffer(void *buf)
{
#if defined(MUX) || defined(MUSH)
    free_sbuf(buf);
#endif

#ifdef PENN
    pse_free(buf);
#endif
}

/*
 * Platform independant function to check the validity of an object.
 */
int ValidObject(dbref obj)
{
#if defined(MUX) || defined(MUSH)
    return ((obj >= 0) && (obj < mudstate.db_top) &&
	    ((db[obj].flags & TYPE_MASK) < NOTYPE)) ? 1 : 0;
#endif

#ifdef PENN
    return (GoodObject(obj) && !Destroyed(obj)) ? 1 : 0;
#endif
}

MUDFUNCTION(fun_spacecall)
{
    MUDPARAM mp;
    
#ifdef MUX
    mp.buff = buff;
    mp.bufc = bufc;
    spacecall((void*) &mp, player, cause, fargs, nfargs);
#endif

#ifdef MUSH
    mp.buff = buff;
    spacecall((void*) &mp, player, cause, fargs, nfargs);
#endif

#ifdef PENN
    mp.buff = buff;
    mp.bufc = bufc;
    mp.pe_info = pe_info;
    spacecall((void*) &mp, executor, enactor, args, nargs);
#endif
}

extern void WriteMudFunctionBuffer(void *mud, const char *str)
{
    MUDPARAM *mp = (MUDPARAM*) mud;

#ifdef MUX
    safe_str(str, mp->buff, mp->bufc);
#endif

#ifdef MUSH
    strcpy(mp->buff, str);
#endif

#ifdef PENN
    safe_str(str, mp->buff, mp->bufc);
#endif
}

extern void FWriteMudFunctionBuffer(void *mud, const char *fmt, ...)
{
    static char buff[20000];
    MUDPARAM *mp = (MUDPARAM*) mud;
    va_list ap;
    va_start(ap, fmt);
    vsprintf(buff, fmt, ap);
    va_end(ap);
    buff[MAX_ATTRIBUTE_LEN-1] = '\0';
    
#if defined(MUX) || defined(PENN)
    safe_str(buff, mp->buff, mp->bufc);
#endif

#ifdef MUSH
    strcpy(mp->buff, buff);
#endif
}

const char *getObjectName(dbref obj)
{
#if defined(MUX) || defined(MUSH)
    return getAttrByNumber(obj, A_NAME);
#endif

#ifdef PENN
    return Name(obj);
#endif
}

dbref parseDbref(const char *s)
{
     const char *p;
     int x;

     if (strlen(s) > 1) {
	  for (p = s+1; *p; p++) {
	       if (!isdigit(*p))
		    return PSE_NOTHING;
	  }
     }
     else
	  return PSE_NOTHING;

     x = atoi(s+1);
     return ((x >= 0) ? x : PSE_NOTHING);
}

dbref getLocation(dbref obj)
{
    return (db[obj].location);
}

dbref getDebugCharacter()
{
    /* Since 0.9.10, we use a variable set by spacecall */
    return debug_char;
}

/*
 * Return 1 if the object is permitted to use spacecalls.
 * GOD must be permitted spacecalls, as messages and events
 * are executed in that context.
 */
int isSpace(dbref obj)
{
    return (Space(obj) || (obj == GOD)) ? 1 : 0;
}


/* 
 * pse_malloc, pse_calloc, pse_free, pse_realloc:
 * Safe memory allocation/deallocation routines.
 */

void *pse_malloc(int size)
{
    void *ptr;
    
    if ( (ptr = malloc(size)) == NULL) {
	abort();
	return(NULL);
    }
    else
	return(ptr);
    
}

void *pse_calloc(int num, int size)
{
    void *ptr;
    
    if ( (ptr = calloc(num, size)) == NULL) {
	abort();
	return(NULL);
    }
    else
	return(ptr);
}

void *pse_realloc(void *ptr, int nsize)
{
    void *nptr;
    
    if ( (nptr = realloc(ptr, nsize)) == NULL) {
	abort();
	return(NULL);
    }
    else
	return(nptr);
}
