/* v0.9
 *
 * varray.c:  Virtual Array (VArray) Code.
 *
 * This program is free software and may be freely redistributed as
 * specified in the GNU General Public License.  Please see the file
 * 'COPYING' for details.
 */

#include <string.h>
#include <stdlib.h>
#include "spaceconf.h"
#include "pseint.h"
#include "varray.h"

char *store_name(VArrayIndex *, const char *);

VArrayIndex *vaAllocateIndex(void)
{    
     VArrayIndex *newidx;

     newidx = (VArrayIndex *) pse_malloc(sizeof(VArrayIndex));
     newidx->root = NULL;
     newidx->htab = NULL;
     newidx->maxindex = -1;
     newidx->strings = NULL;
     newidx->strsize = 0;
     
     return(newidx);
}

void vaFreeIndex(VArrayIndex *vidx)
{
     VAIDXENTRY *ptr, *tmp;

     for (ptr = vidx->root; ptr != NULL; ) {

	  tmp = ptr;
	  ptr = ptr->next;
	  pse_free(tmp);

     }

     hashFree(vidx->htab);
     pse_free(vidx->strings);
     
}

VArray *vaAllocateVArray(int type, int size, int num)
{
     VArray *newva;
     int i;

     newva = (VArray *) pse_malloc(sizeof(VArray));

     newva->type = type;

     newva->nitems = num;
     newva->index = (char **) pse_calloc(num, sizeof(void *));

     for (i = 0; i < num; i++)
	  newva->index[i] = NULL;

     newva->mark = 0;
     newva->size = size;
     newva->data = (char *) pse_malloc(size);

     return(newva);
}

void vaFreeVArray(VArray *va)
{

     pse_free(va->data);
     pse_free(va->index);
     pse_free(va);

}

/* Store a name in the index string block */
char *store_name(VArrayIndex *vidx, const char *str)
{
     int len;
     char *ret = NULL;

     len = strlen(str);

     if (vidx->strings == NULL) {
	  vidx->strings = (char *) pse_malloc(VA_STRINGCHUNK);
	  vidx->strmark = 0;
	  vidx->strsize = VA_STRINGCHUNK;
     }

     while ((vidx->strsize - vidx->strmark) < len + 1) {
	  vidx->strsize += VA_STRINGCHUNK;
	  vidx->strings = (char *) pse_realloc(vidx->strings, vidx->strsize);
     }

     ret = vidx->strings + vidx->strmark;
     strcpy(ret, str);
     vidx->strmark += len + 1;
     return(ret);
	  
}

VArrayIndex *vaCreateIndexFromTable(VAIDXENTRY *table)
{
     int i;
     VArrayIndex *vidx;

     vidx = vaAllocateIndex();

     for (i = 0; *(table[i].name) != '\0'; i++) {
	  
	  /* Assign an index, see if we can find the DB attribute number
	     and add the table entry into the hash table. */
	  vidx->maxindex = table[i].index > vidx->maxindex ?
	       table[i].index : vidx->maxindex;
	  table[i].dbattr = getAttrNumber(table[i].name);
	  hashInsert(table[i].name, (void *)&(table[i]), &(vidx->htab));

     }

     return(vidx);

}

void vaAddIndexEntry(VArrayIndex *vidx, char *name, int flags)
{
     VAIDXENTRY *newent, *ptr;

     newent = (VAIDXENTRY *) pse_malloc(sizeof(VAIDXENTRY));

     newent->name = store_name(vidx, name);
     newent->flags = flags;
     newent->dbattr = getAttrNumber(name);
     newent->index = ++(vidx->maxindex);
     newent->next = NULL;

     if (vidx->root == NULL) 
	  vidx->root = newent;

     else {

	  for (ptr = vidx->root; ptr->next != NULL; ptr = ptr->next)
	       ;

	  ptr->next = newent;

     }

     hashInsert(name, (void *)newent, &(vidx->htab));

}

int vaGetMaxIndex(VArrayIndex *vidx)
{
     return (vidx->maxindex);
}

int vaGetIndex(VArrayIndex *vidx, const char *search)
{
     VAIDXENTRY *found;
     int ret;

     found = (VAIDXENTRY *) hashSearch(search, vidx->htab);

     if (found == NULL)
	  ret = -1;

     else
	  ret = found->index;

     return (ret);

}

int vaSetItem(VArray *va, int newindex, const void *newdata)
{
     int dsize;


     switch (va->type) {
     
     case VA_NUMBER:
	  dsize = 4;
	  break;

     default:
	  dsize = strlen((const char *) newdata) + 1;
	  break;

     }

     if (newindex >= va->nitems)
	  return (1);		/* Invalid index. */

     if (va->index[newindex] != NULL) {

	  if (va->type != VA_NUMBER)
	       return (1);	/* Can't re-set string or formula. */
	  else
	       memcpy(va->index[newindex], newdata, dsize);

     }
     else if (va->size - va->mark > dsize) {

	  va->index[newindex] = va->data + va->mark;
	  memcpy(va->data + va->mark, newdata, dsize);
	  va->mark += dsize;

     }
     else
	  return (1);		/* Out of space. */

     return(0);			/* Success! */
	  
}

float vaGetItemAsFloat(VArray *va, int idx)
{
     void *found = NULL;

     if (idx < va->nitems && va->type == VA_NUMBER)
	  found = va->index[idx];

     if (found != NULL)
	  return ((float) *((float *) found));
     else
	  return (0.0);
	  
}

int vaGetItemAsInt(VArray *va, int idx)
{
     void *found = NULL;

     if (idx < va->nitems && va->type == VA_NUMBER)
	  found = va->index[idx];

     if (found != NULL)
	  return ((int) *((int *) found));
     else
	  return (0);
	  
}

char *vaGetItem(VArray *va, int idx)
{
     void *found = NULL;

     if (idx < va->nitems)
	  found = va->index[idx];

     return ((char *) found);
     
}
