/* v1.1
 *
 * class.h:  Class/Record Module Header.
 *
 * This program is free software and may be freely redistributed as
 * specified in the GNU General Public License.  Please see the file
 * 'COPYING' for details.
 */

#define NAME_LEN 32

/*
 * Field
 */

/* Field flags */
#define FF_WRITE_BACK  0x00000001 /* bit 0 - dirty bit, item has been changed */
#define FF_EXISTS      0x00000002 /* bit 1 - field exists */
#define FF_FREE_ME     0x00000004 /* bit 2 - free pval upon deletion */

typedef struct field_struct {
    int flags;
    union {
	int ival;
	float fval;
	char *pval;
    } u;
} FIELD;

/*
 * Record definition
 */

typedef struct class_struct Class; /* fwd definition */
typedef struct record_struct Record; /* fwd definition */

struct record_struct {
    char name[NAME_LEN];
    Class* class;
    dbref db_obj;
    FIELD *fields;
    Record *queue; /* Used during validating/adding record */
    void *ref;  /* Class-specific pointer. */

    Record *next;
    Record *prev;

};

/*
 * Class
 */

/* Field definition flags */
#define FD_REQUIRED 0x00000001	/* bit 0 */
#define FD_TYPEMASK 0x0000001e  /* bits 1-4 */
#define FD_OPTMASK  0x000000e0	/* bits 5-7 */

/* Options --  bits that select options for particular Field types. */
#define FD_Type(x) (x << 1)
#define FD_TypeOf(x) (x & FD_TYPEMASK)
#define FD_Option(x) (x << 5)
#define FD_OptionOf(x) (x & FD_OPTMASK)

/* Number options */
#define FD_TYPE_INT FD_Type(1)
#define FD_TYPE_FLOAT FD_Type(2)
#define FD_TYPE_STRING FD_Type(3)

/* String options */
#define FD_NOPARSE FD_Option(1)     /* String shouldn't be parsed. */
#define FD_FASTPARSE FD_Option(2)   /* String should be parsed minimally (but quickly) */
#define FD_FULLPARSE FD_Option(3)   /* String should be fully parsed using the MU* parser */

/* Field definition */
typedef struct fielddef_struct {
    int index;
    const char *attr;
    int flags;
} FieldDef;

/* Class flags */
#define CLS_PUBLIC  0x00000001  /* Can be instantiated with the add_object spacecall. */

struct class_struct {
    char name[NAME_LEN];
    int flags;
    int nfields;
    const FieldDef **defs;
    Record *record_head;
    Record *record_tail;

    /* Class handlers. */
    void (*on_add)(Record *);
    int (*validate)(Record *); /* Return 0 if OK, 1 if not. */
    void (*on_reload)(Record *);
    int (*on_remove)(Record *);

    Class *parent;
    Class *next;
};

/*
 * Class/Record Operations
 */

void clsInit();
void clsShutdown();
Class *clsRegisterClass(const char *name, Class *parent, int flags, 
			const FieldDef *def_table, 
			int (*validate_fun)(Record *),
			void (*add_fun)(Record *), 
			void (*reload_fun)(Record *), 
			int (*remove_fun)(Record *));
Class *clsClassOf(dbref record);
int clsRecordExists(Class *class, dbref record);
int clsAddRecord(dbref player, dbref record);
Record *clsLoadAndValidate(Class *class, dbref record);
void clsQueue(Record *parent, Record *child);
Record *clsRetrieve(Record *parent);
void clsAddRecordDirect(Record *recptr);
int clsReloadField(Record *recptr, int index);
#ifdef NOT_IMPLEMENTED_YET
void clsReloadRecordDirect(dbref player, Class *class, Record *oldrecptr);
#endif
Record *clsFindRecord(Class *class, dbref record);
int clsRemoveRecord(Record *recptr);
void clsWriteRecord(Record *recptr);
void clsSetString(Record *record, int index, const char *value);
int clsValidateDbref(const char *dbrefstr);
int clsValidateList(const char *list, int (*validate_item)(const char *item));
int clsValidateInt(const char *str);
int clsValidateFloat(const char *str);

/* Basic field operations */
#define F_INT(R, F) R->fields[F].u.ival
#define F_FLOAT(R, F) R->fields[F].u.fval
#define F_PTR(R, F) R->fields[F].u.pval
#define F_SET_WRITE_BACK(R, F) R->fields[F].flags |= FF_WRITE_BACK
#define F_EXISTS(R, F) ((R->fields[F].flags & FF_EXISTS) > 0)
