/* v1.1
 *
 * eng.c:  Ship engineering code.
 *
 * This program is free software and may be freely redistributed as
 * specified in the GNU General Public License.  Please see the file
 * 'COPYING' for details.
 */

#include "spaceconf.h"
#include "pseint.h"
#include "dbint.h"
#include "space.h"
#include "class.h"
#include "object.h"
#include "tactical.h"
#include "nav.h"
#include "eng.h"
#include "damage.h"
#include "smisc.h"
#include "events.h"
#include "shields.h"

void engTacticalTurn(TAG *object)
{
     char buff[SMALL_BUF_SIZE];
     SHIP *ship = object->shipdata;
     int time_left;
     int engineering = dbrefEngine(ship);
     int overload_change;

     /* Charge batteries */
     ship->battery_level += ship->alloc_batt;

     /* deplete battery */
     ship->battery_level -= ship->battery_discharge;

     /* bleed batteries */
     if (ship->battery_status == BTTY_ONLINE)
	  ship->battery_level *= BATTERY_BLEED;

     /* check for bettery overcharge */
     if (ship->battery_level > F_INT(ship->rec, battery_capacity))
	  ship->battery_level = F_INT(ship->rec, battery_capacity);

     /* force a reallocation if batts are too low */
     if (ship-> battery_discharge > ship->battery_level)
	  engAllocCheck(object);

     /* See if we still have power to maintain life support */

     if (ship->reactor_output < 
	 (int)(0.05 * F_INT(ship->rec, reactor_output_max)))
	  evTrigger(object, EVENT_LOW_POWER, "");

     if (ship->reactor_setting < F_INT(ship->rec, reactor_stress_level) &&
	 ship->overload_points > 0) {

	  /* we're losing overload points.  Whee! */
	  ship->overload_points -= ((F_INT(ship->rec, reactor_stress_level) -
				     ship->reactor_setting) / 2);

	  if (ship->overload_points < 0)
	       ship->overload_points = 0;
     }

     /* If there's no reactor stress, we're done. */
     if (ship->reactor_setting <= F_INT(ship->rec, reactor_stress_level))
	  return;

     /* Otherwise, we're accruing overload points */
     overload_change = ship->reactor_setting - 
	 F_INT(ship->rec, reactor_stress_level);

     if (ship->reactor_setting > 100)
	  overload_change += ((F_INT(ship->rec, reactor_overload_penalty)) *
			      (ship->reactor_setting - 100));

     /* add them in and flash a message */
     ship->overload_points += overload_change;

     if (FRAND < ((float) ship->overload_points / 
		  F_INT(ship->rec, max_overload_points) - 1)) {

	  /* damage the engine 1-3 points. If function returns a 1,
	   * we just blew up, and object is not valid any more 
	   * so return. The 'Destroyed' flag will have been set,
	   * so the linked list isn't disrupted.
	   */
	  if (damDamageEngine(object, (int)(FRAND * 2 + 1)) == 1)
	       return;
     }

     /* calculate the time left until crystal stress at this pace */

     time_left = (int)(6.0 * (F_INT(ship->rec, max_overload_points) -
			      ship->overload_points) / overload_change);

     if (time_left < 6 && ship->eng_safeties) {
	  /* force a reactor slowdown */

	  MFNotify(ship, -1, CONS_ENG | CONS_DAM, CONS_ACTIVE,
		   "[Reactor at critical stress. Emergency override "
		   "initiated]");
	  engSetReactorOutput(object, PSE_NOTHING, 
			      F_INT(ship->rec, reactor_stress_level));
     }
     else if (time_left < 61 && ship->eng_warnings) {

	  if (time_left > 0)
	       sprintf(buff, "[Engine overstressing.  Reactor stress in"
		       " %d seconds]", time_left);
	  else
	       sprintf(buff, "[Engine is overstressed. Stress at %d%%]",
		       100 * object->shipdata->overload_points /
		       F_INT(object->shipdata->rec, max_overload_points));

	  NotifyExcept(engineering, PSE_NOTHING, buff);
     }

     return;
}

void engSetReactorOutput(TAG *object, dbref player, int setting)
{
     SHIP *ship = object->shipdata;

     if (setting < 5) {
	  Notify(player, "The minimum output level to maintain life "
		 "support is 5%.");
	  return;
     }
     else if (setting > F_INT(ship->rec, reactor_setting_limit)) {
	  FNotify(player, "The maximum output level is %d%%.",
		  F_INT(ship->rec, reactor_setting_limit));
	  return;
     }
     else
     {
	  ship->reactor_setting = setting;

	  ship->reactor_output = (int)((ship->current_reactor_output_max
					- ship->reactor_drain) * 
				       setting / 100);

	  MFNotify(ship, player, CONS_ENG, CONS_ACTIVE,
		   "[Reactor output level set to %d%%]\n"
		   "[Power available: %d]",
		   setting, ship->reactor_output);

	  /* overallocation check */
	  engAllocCheck(object);
	  return;
     }

}

void engAllocCheck(TAG *object)
{
     SHIP *ship = object->shipdata;
     int allocation;
     int available_power;

     /* recalculate reactor_output */
     ship->reactor_output = (int)((ship->current_reactor_output_max -
				   ship->reactor_drain) * 
				  ship->reactor_setting / 100);

     available_power = ship->reactor_output;
     if (ship->battery_status == BTTY_ONLINE) {
	  if (ship->battery_level < F_INT(ship->rec, battery_discharge_max))
	       available_power += ship->battery_level;
	  else
	       available_power += F_INT(ship->rec, battery_discharge_max);
     }

     allocation = ship->alloc_tac + ship->alloc_nav + ship->alloc_shields +
	  ship->alloc_batt;

     if (allocation > available_power) {

	  MFNotify(ship, -1, CONS_ENG, CONS_ACTIVE,
		   "[Drop in net power.  Global energy reallocation "
		   "performed]");
	  engAllocate(object, PSE_NOTHING, ship->alloc_tac, ship->alloc_nav,
		      ship->alloc_shields, ship->alloc_batt);

	  navAllocCheck(object);
	  shdAllocCheck(object);
     }
     else {
	  if (allocation > ship->reactor_output)
	       ship->battery_discharge = allocation - 
		    ship->reactor_output;
	  else
	       ship->battery_discharge = 0;
     }

     return;
}

void engAllocate(TAG *object, dbref player, int tac, int nav, int shields, 
		 int batt)
{

     SHIP *ship = object->shipdata;
     int available_power;
     float total_allocation;
     float power_ratio;   /* fraction the the allocation to receive *
			   * in the case of over-allocation.        */
     int old_nav, old_tac, old_shields;

     /* calculate max available power */
     available_power = ship->reactor_output;
     if (ship->battery_status == BTTY_ONLINE) {
	  if (F_INT(ship->rec, battery_discharge_max) <= ship->battery_level)
	       available_power += F_INT(ship->rec, battery_discharge_max);
	  else
	       available_power += ship->battery_level;
     }

     /* if disabled, no power is available */
     if (Disabled(ship)) {
	  Notify(player, "Ship is disabled.  Cannot allocate power.");
	  available_power = 0;
     }

     /* set negative allocations to zero */
     if (tac < 0)  tac = 0;
     if (nav < 0) nav = 0;
     if (shields < 0) shields = 0;
     if (batt < 0) batt = 0;

     /* calculate total allocation (a float to avoid overflow) */
     total_allocation = (float) tac + (float) nav + (float) shields +
	 (float) batt;

     /* if allocation > power, apply cuts */
     if (total_allocation > available_power) {

	  power_ratio = (available_power / total_allocation);
	  tac = (power_ratio * tac);
	  nav = (power_ratio * nav);
	  shields = (power_ratio * shields);
	  batt = (power_ratio * batt);

     }

     /* save the old values */
     old_nav = ship->alloc_nav;
     old_tac = ship->alloc_tac;
     old_shields = ship->alloc_shields;

     /* assign the new values */
     ship->alloc_tac = tac;
     ship->alloc_nav = nav;
     ship->alloc_shields = shields;
     ship->alloc_batt = batt;

     /* update battery discharge */
     ship->battery_discharge = tac + nav + shields + batt - 
	  ship->reactor_output;
     if (ship->battery_discharge < 0)
	  ship->battery_discharge = 0;

     /* notify the player */
     MFNotify(ship, player, CONS_ENG, CONS_ACTIVE,
	      "[Total available power: %d]\n[Allocated to Tactical: %d  "
	      "Nav: %d  Shields: %d  Batteries: %d]", available_power, tac, 
	      nav, shields, batt);

     /* check for shortfalls at other stations */
     if (old_nav > nav) {

	  if (object->tractor_source != NULL)
	       MFNotify(object->tractor_source->shipdata, -1, CONS_TAC, 
			CONS_ACTIVE,  "[Tractor beam target has decreased "
			"power to engines]");

	  navAllocCheck(object);
     }

     if (old_tac > tac)
	  tacAllocCheck(object);

     if (old_shields > shields) {
	  shdAllocCheck(object);
	  log_space("TRACE: shield reallocation check 1.");
     }

     /* check for a power increase at other stations */
     if (old_tac < tac) 
	  MFNotify(ship, -1, CONS_TAC, CONS_ACTIVE,
		   "[New tactical allocation is %d]", tac);

     if (old_nav < nav) {
	  MFNotify(ship, -1, CONS_NAV, CONS_ACTIVE, "[New nav allocation is "
		   "%d]", nav);

	  if (object->tractor_source != NULL)
	       MFNotify(object->tractor_source->shipdata, -1, CONS_TAC, 
			CONS_ACTIVE,
			"[Tractor beam target has increased their power to "
			"engines]");
     }

     if (old_shields < shields) 
	  MFNotify(ship, -1, CONS_SHD, CONS_ACTIVE,
		   "[New shield allocation is %d]", shields);

     return;
}

void engBatteryOn(TAG *object, dbref player)
{

     SHIP *ship = object->shipdata;

     /* active.  Check battery status */
     switch (ship->battery_status) {
     case BTTY_ONLINE:
	  Notify(player, "Batteries are already online.");
	  break;

     case BTTY_DAMAGED:
	  Notify(player, "Batteries are damaged and may not be brought"
		 " online.");
	  break;

     case BTTY_OFFLINE:
	  ship->battery_status = BTTY_ONLINE;
	  MFNotify(ship, player, CONS_ENG, CONS_ACTIVE, "[Batteries now "
		   "online]");
	  break;
     }

     return;

}

void engBatteryOff(TAG *object, dbref player)
{

     SHIP *ship = object->shipdata;

     switch (ship->battery_status) {
     case BTTY_ONLINE:
	  ship->battery_status = BTTY_OFFLINE;

	  MFNotify(ship, player, CONS_ENG, CONS_ACTIVE, "[Batteries are now "
		   "offline]");

	  /* check for overallocation */
	  engAllocCheck(object);
	  break;

     case BTTY_DAMAGED:
	  Notify(player, "Batteries are damaged and need not be "
		 "taken off line.");
	  break;

     case BTTY_OFFLINE:
	  Notify(player, "Batteries are already offline.");
	  break;
     }

     return;

}

void engShutdownReactor(TAG *object, dbref player)
{
	if (space_info[object->space].flags & SPACE_LOGGED)	

	  Notify(player, "This command is not available in real space.");

     else {

	  MFNotify(object->shipdata, player, CONS_ENG, CONS_ACTIVE,
		   "[Reactor shutdown]");
	  clsRemoveRecord(object->rec);

     }

     return;
}

void engStartReactor(TAG *object, dbref player, dbref data)
{
    char *buff;
    TAG *newship;
    int space;
    
    buff = pse_malloc(MAX_ATTRIBUTE_LEN);
    getAttrByNameLen(buff, data, STATUS_SPACE, MAX_ATTRIBUTE_LEN);
    space = atoi(buff);
    pse_free(buff);
    
    if (space_info[space].flags & SPACE_LOGGED) {
	Notify(player, "This command is not available in real space.");
	return;
    }
    
    if (space_info[space].flags & SPACE_LOCKED) {
	Notify(player, "Space is temporarily locked.  Please try again "
	       "later.");
	return;
    }
    
    if (clsAddRecord(player, data))
	return;
    
    if ( (newship = objFindObject(data)) == NULL) {
	Notify(player, "Unable to activate ship.  Please contact an"
	       " admin.");
	return;
    }
    
    if (Ship(newship))
	MFNotify(objFindObject(data)->shipdata, player, CONS_ENG, 
		 CONS_ACTIVE, "[Reactor started and set to 5%% output]");
    else
	Notify(player, "[Reactor started and set to 5%% output]");
    
    return;
}

void engSetReactor(TAG *object, dbref player, int setting)
{

     SHIP *ship = object->shipdata;

     if (setting < 5) {
	  Notify(player, "The minimum output level is 5%.");
	  return;
     }
     else if (setting > F_INT(ship->rec, reactor_setting_limit)) {
	  FNotify(player, "The maximum output level is %d%%.",
		  F_INT(ship->rec, reactor_setting_limit));
	  return;
     }
     else {
	  ship->reactor_setting = setting;

	  ship->reactor_output = (int)(ship->current_reactor_output_max
				       * setting / 100);

	  MFNotify(ship, player, CONS_ENG, CONS_ACTIVE,
		   "[Reactor output level set to %d%%]\n[Power available: %d]",
		   setting, ship->reactor_output);

	  /* overallocation check */
	  engAllocCheck(object);
	  return;
     }
}
