/* v1.1
 *
 * object.c:  Space object code.
 *
 * This program is free software and may be freely redistributed as
 * specified in the GNU General Public License.  Please see the file
 * 'COPYING' for details.
 */

#include <assert.h>
#include <stdlib.h>
#include <ctype.h>

#include "btree.h"
#include "spaceconf.h"
#include "pseint.h"
#include "dbint.h"
#include "space.h"
#include "class.h"
#include "output.h"
#include "object.h"
#include "damage.h"
#include "smisc.h"
#include "shields.h"
#include "events.h"
#include "nav.h"
#include "sensors.h"
#include "flag.h"
#include "rxl/rxl.h"

#ifdef ENABLE_SHIELD_CLASSES
#include "scm/scm.h"
#endif

static const FieldDef object_field_defs[] = {
    {size, "CF_SIZE", FD_TYPE_FLOAT|FD_REQUIRED},
    {xport_range, "CF_XPORT_RANGE", FD_TYPE_INT|FD_REQUIRED},
    {sensor_range, "CF_SENSOR_RANGE", FD_TYPE_INT|FD_REQUIRED},
    {cloak_effect, "CF_CLOAK_EFFECT", FD_TYPE_FLOAT},
    {critical_range, "CF_CRITICAL_RANGE", FD_TYPE_INT},
    {space, "ST_SPACE", FD_TYPE_INT|FD_REQUIRED},
    {startpos, "ST_STARTPOS", FD_TYPE_STRING|FD_REQUIRED},
    {flags, "ST_FLAGS", FD_TYPE_STRING|FD_REQUIRED},
    {range_name, "CF_RANGE_NAME", FD_TYPE_STRING|FD_REQUIRED},
    {range_factor, "CF_RANGE_FACTOR", FD_TYPE_FLOAT},
    {rxl_name, "CF_RANGE_TRANSLATOR", FD_TYPE_STRING},
    {-1, "", 0}};
 
static const FieldDef ship_field_defs[] = {
    {shipclass, "CF_SHIPCLASS", FD_TYPE_STRING|FD_REQUIRED},
    {type, "CF_TYPE", FD_TYPE_STRING|FD_REQUIRED},
    {owner_name, "CF_OWNER_NAME", FD_TYPE_STRING|FD_REQUIRED},
    {owner_num, "CF_OWNER_NUM", FD_TYPE_INT|FD_REQUIRED},
    {hull, "CF_HULL", FD_TYPE_INT|FD_REQUIRED},
    {tractor_present, "CF_TRACTOR_STATUS", FD_TYPE_INT},
    {tractor_effect, "CF_TRACTOR_EFFECT", FD_TYPE_FLOAT},
    {door_present, "CF_DOOR_STATUS", FD_TYPE_INT},
    {door_size, "CF_DOOR_SIZE", FD_TYPE_FLOAT},
    {door_side, "CF_DOOR_SIDE", FD_TYPE_INT},
    {reactor_output_max, "CF_REACTOR_OUTPUT", FD_TYPE_INT|FD_REQUIRED},
    {max_overload_points, "CF_REACTOR_OVER_POINTS", FD_TYPE_INT|FD_REQUIRED},
    {reactor_setting_limit, "CF_REACTOR_LEVEL_STRESS", FD_TYPE_INT|FD_REQUIRED},
    {reactor_stress_level, "CF_REACTOR_LEVEL_NORMAL", FD_TYPE_INT|FD_REQUIRED}, 
    {reactor_overload_penalty, "CF_REACTOR_OVER_PENALTY", FD_TYPE_INT|FD_REQUIRED},
    {battery_capacity, "CF_BATTS_CAPACITY", FD_TYPE_INT|FD_REQUIRED},
    {battery_discharge_max, "CF_BATTS_DISCHARGE", FD_TYPE_INT|FD_REQUIRED},
    {warp_factor, "CF_WARP_FACTOR", FD_TYPE_FLOAT|FD_REQUIRED},
    {warp_rated_max, "CF_WARP_MAX", FD_TYPE_FLOAT|FD_REQUIRED},
    {warp_accel, "CF_WARP_ACCEL", FD_TYPE_FLOAT|FD_REQUIRED},
    {cloak_present, "CF_CLOAK_STATUS", FD_TYPE_INT},
    {cloak_cost, "CF_CLOAK_COST", FD_TYPE_INT},
    {scanner_range, "CF_SCANNER_RANGE", FD_TYPE_INT|FD_REQUIRED},
    {transmitter_range, "CF_COMM_RANGE", FD_TYPE_FLOAT|FD_REQUIRED},
    {receiver_sensitivity, "CF_COMM_SENSITIVITY", FD_TYPE_FLOAT|FD_REQUIRED},
    {shield_config, "CF_SHIELD_CONFIG", FD_TYPE_INT|FD_REQUIRED},
    {shield_factor, "CF_SHIELD_FACTOR", FD_TYPE_FLOAT|FD_REQUIRED},
    {shield_power, "CF_SHIELD_POWER", FD_TYPE_INT|FD_REQUIRED},
    {shield_max_charge, "CF_SHIELD_CHARGE_RATE", FD_TYPE_INT|FD_REQUIRED},
    {gun_string, "CF_GUN_NAME", FD_TYPE_STRING},
    {gun_qty, "CF_GUN_QTY", FD_TYPE_INT|FD_REQUIRED},
    {gun_power, "CF_GUN_POWER", FD_TYPE_INT},
    {gun_range, "CF_GUN_RANGE", FD_TYPE_INT},
    {gun_range_percent, "CF_GUN_DELIVERY", FD_TYPE_FLOAT},
    {gun_charge_rate, "CF_GUN_CHARGE_RATE", FD_TYPE_INT},
    {gun_arc, "CF_GUN_ARC", FD_TYPE_INT},
    {torp_string, "CF_TORP_NAME", FD_TYPE_STRING},
    {torp_qty, "CF_TORP_QTY", FD_TYPE_INT|FD_REQUIRED},
    {torp_loaders, "CF_TORP_LOADERS", FD_TYPE_INT},
    {torp_ammo, "CF_TORP_AMMO", FD_TYPE_INT},
    {torp_power, "CF_TORP_POWER", FD_TYPE_INT},
    {torp_range, "CF_TORP_RANGE", FD_TYPE_INT},
    {torp_accuracy, "CF_TORP_ACCURACY", FD_TYPE_FLOAT},
    {torp_charge_rate, "CF_TORP_CHARGE_RATE", FD_TYPE_INT},
    {damcon_teams, "CF_DAMCON_TEAMS", FD_TYPE_INT|FD_REQUIRED},
    {hull_status, "ST_HULL_DAMAGE", FD_TYPE_STRING|FD_REQUIRED},
    {system_status, "ST_SYSTEM_DAMAGE", FD_TYPE_STRING|FD_REQUIRED},
    {shipflags, "ST_SHIPFLAGS", FD_TYPE_STRING|FD_REQUIRED},
    {docked_at, "ST_DOCKED_AT", FD_TYPE_STRING},
    {dbref_bridge, "CF_DBREF_BRIDGE", FD_TYPE_STRING|FD_REQUIRED},
    {dbref_eng, "CF_DBREF_ENG", FD_TYPE_STRING|FD_REQUIRED},
    {turn_factor, "CF_TURN_FACTOR", FD_TYPE_FLOAT|FD_REQUIRED},
    {roll_factor, "CF_ROLL_FACTOR", FD_TYPE_FLOAT|FD_REQUIRED},
#ifdef ENABLE_ODOMETER
    {odometer, "ST_ODOMETER", FD_TYPE_INT},
#endif
#ifdef ENABLE_TIME_TRACKER
    {time_active, "ST_TIME_ACTIVE", FD_TYPE_INT},
#endif
    {consoles, "CF_CONSOLE_LIST", FD_TYPE_STRING|FD_REQUIRED},
    {-1, "", 0}};

Class *vehicleclass;

int objValidate(Record *recptr);
void objAdd(Record *recptr);
/*void objReload(Record *recptr);*/
int objRemove(Record *recptr);    
int shipValidate(Record *recptr);
void shipAdd(Record *recptr);
int shipRemove(Record *recptr);
int validate_console(const char *);

void objInit(void)
{
    /* Register public class 'Object' */
    clsRegisterClass("Object", NULL, CLS_PUBLIC, object_field_defs, 
		     objValidate, objAdd, NULL /*objReload*/, objRemove);
    /* Register internal class 'Vehicle' */
    vehicleclass = clsRegisterClass("Vehicle", NULL, 0, ship_field_defs,
				    shipValidate, shipAdd, NULL, shipRemove); 
    return;
}

int objValidate(Record *recptr)
{
    unsigned int fl;
    Record *vptr;

    if (F_INT(recptr, space) < 0 || F_INT(recptr, space) >= NUM_SPACES) {
	log_space("#%d:  Bad space number.  Valid range is 0 to %d.", 
		  recptr->db_obj, NUM_SPACES - 1);
	return 1;
    }
    
    if (loadSymbolicFlags(&fl, FLIST_OBJECT, F_PTR(recptr, flags)) != NULL) {	
	log_space("#%d: Non-symbolic object flags found. Trying old-style.",
		  recptr->db_obj);
	if (loadBinaryFlags(&fl, F_PTR(recptr, flags)) == 0) {
	    log_space("#%d: Not binary flags either. Giving up.", recptr->db_obj);
	    return 1;
	}
	
	log_space("#%d: Non-symbolic object flags updated [%s].",
		  recptr->db_obj, F_PTR(recptr, flags));
	clsSetString(recptr, flags, saveSymbolicFlags(flags, FLIST_OBJECT, 0));
	F_SET_WRITE_BACK(recptr, flags);
    }

#ifdef ENABLE_FLOATS
    if (clsValidateList(F_PTR(recptr, startpos), clsValidateFloat)) {
	log_space("#%d: Invalid ST_STARTPOS attribute.", recptr->db_obj);
	return 1;
    }
#else
    if (clsValidateList(F_PTR(recptr, startpos), clsValidateInt)) {
	log_space("#%d: Invalid ST_STARTPOS attribute.", recptr->db_obj);
	return 1;
    }
#endif

    if (F_FLOAT(recptr, size) <= 0) {
	log_space("#%d: CF_SIZE must be greater than zero.", recptr->db_obj);
	return 1;
    }

    if (F_INT(recptr, sensor_range) <= 0) {
	log_space("#%d: CF_SENSOR_RANGE must be greater than zero.", recptr->db_obj);
	return 1;
    }

    if (rxlLookup(F_PTR(recptr, rxl_name)) == -1) {
	log_space("#%d: CF_RANGE_TRANSLATOR plugin (%s) was not found.", \
		  recptr->db_obj, F_PTR(recptr, rxl_name));
	return 1;
    }

    /* If object is a vehicle, load and validate vehicle data. */
    if ((fl & TYPE_SHIP) != 0) {
	if ( (vptr = clsLoadAndValidate(vehicleclass, recptr->db_obj)) 
	     == NULL) 
	    return 1;

	/* Add vehicle to queue so we can add it later. */
	clsQueue(recptr, vptr);
    }

    return 0;

}

void objAdd(Record *recptr)
{
    TAG *new_object;
    const char *badflag;
    Record *vptr;

    new_object = (TAG *)pse_malloc(sizeof(TAG));
   
    /* Initialize the new object */

    new_object->rec = recptr;
    recptr->ref = (void *) new_object;

    new_object->space = F_INT(recptr, space);
    new_object->tagflags = 0;
    badflag = loadSymbolicFlags(&(new_object->tagflags), FLIST_OBJECT,
			  F_PTR(recptr, flags));
    assert(badflag == NULL);

    fillXYZCoords(F_PTR(recptr, startpos), &(new_object->dc_pos));
    
    new_object->range = HubRange(new_object->dc_pos);
    
    new_object->heading.bearing = 0.0;
    new_object->heading.elevation = 0.0;
    new_object->heading.range = 1;
    
    new_object->heading_adj.bearing = 0;
    new_object->heading_adj.elevation = 0;
    new_object->heading_adj.range = 0;
    
    new_object->v_move[0] = 1.0;
    new_object->v_move[1] = 0.0;
    new_object->v_move[2] = 0.0;
    
    new_object->speed = 0.0;
    new_object->tractor_source = NULL;
    
    new_object->moveID = move_turn_id;
    new_object->range_prev = NULL;
    new_object->range_next = NULL;
    new_object->huge_prev = NULL;
    new_object->huge_next = NULL;
    
    new_object->roll = 0.0;
    new_object->roll_adj = 0.0;
    
    new_object->pending_lock = NULL;
    new_object->locked_on = NULL;
    new_object->contact_list = NULL;
    
    new_object->rxl_ID = rxlLookup(F_PTR(recptr, rxl_name));

    if (!F_EXISTS(recptr, range_factor) || F_FLOAT(recptr, range_factor) == 0.0)
	F_FLOAT(recptr, range_factor) = 1;

    if (!F_EXISTS(recptr, cloak_effect))
	F_FLOAT(recptr, cloak_effect) = 0.0;
    
    if (!F_EXISTS(recptr, critical_range))
	F_INT(recptr, critical_range) = 0;

    if (Ship(new_object)) {
	vptr = clsRetrieve(recptr);
	assert(vptr != NULL);
	clsAddRecordDirect(vptr);
	new_object->shipdata = (SHIP *) vptr->ref;
	damInitSystemDamage(new_object);
    }
    else
	new_object->shipdata = NULL;
    
    /* Link in new object */
    snsAddObject(new_object, F_INT(recptr, space));
    
    if (space_info[F_INT(recptr, space)].flags & SPACE_LOGGED)
	log_space("Added %s (#%d) to real space.", 
		  new_object->rec->name,
		  recptr->db_obj); 
    
    evTrigger(new_object, EVENT_STARTUP, "i", F_INT(recptr, space));
    
    return;   
}

/* objRemove:  The first time this is called, the object is marked for
 *             removal.  The second time it is called, the object's record
 *             is actually removed.
 */
int objRemove(Record *recptr)
{
    TAG *object;

    object = (TAG *) recptr->ref;

    if (Removed(object)) {
	/* FIX-ME : Save the current state of the object here */
	
	/* Remove ship record if there is one. */
	if (Ship(object))
	    clsRemoveRecord(object->shipdata->rec);

	pse_free(object);
	return 0;
    }
    else {
	/* Queue object for removal. */
	evTrigger(object, EVENT_SHUTDOWN, "");
	object->tagflags |= REMOVED;
	return -1;
    }
}	

int shipRemove(Record *recptr) {
    SHIP *ship = (SHIP *) recptr->ref;

    pse_free(ship);
    return 0;
}

void objUpdateState(TAG *object, void *mud)
{
    XYZ pos;
    char *buff;
    SHIP *ship;
#ifdef ENABLE_SHIELD_CLASSES
    int i;
#endif
    struct timeval now;

    if (!ValidObject(object->rec->db_obj)) {
	
        FWriteFunctionBuffer("#-1 Data object #%d was not found. "
			     "Data not updated.", object->rec->db_obj);
	return;
    }
    
    if (Ship(object)) {	

	ship = object->shipdata;

	damWriteDamage(object->shipdata);
	
#ifdef ENABLE_ODOMETER
	navWriteOdometer(object->rec->db_obj, object->shipdata);
#endif
	
#ifdef ENABLE_TIME_TRACKER
	navWriteTimeActive(object->rec->db_obj, object->shipdata);
#endif
	
	/* Write out the ship data if necessary */
	clsWriteRecord(object->shipdata->rec);

#ifdef ENABLE_SHIELD_CLASSES
	log_space("SCM: Object save: %d", ship->sc_num);
	for (i = 0; i < ship->sc_num; i++) {
	    scmSaveData(object->rec->db_obj, &ship->sc[i]);
	}
#endif
    }

    buff = pse_malloc(MAX_ATTRIBUTE_LEN);

    gettimeofday(&now, NULL);

    object_position(object, &pos, &now, 0);
    
    sprintf(buff, RANGEF " " RANGEF " " RANGEF, pos.x, pos.y, pos.z);
    clsSetString(object->rec, startpos, buff);

    pse_free(buff);

    F_SET_WRITE_BACK(object->rec, startpos);
    clsWriteRecord(object->rec);

    log_space("#%d: Wrote current attributes.", object->rec->db_obj);
    FWriteFunctionBuffer("#%d: Wrote current attributes.",
			 object->rec->db_obj);
}

void objFreeMarkedObjects(int space)
{
    TAG *object, *prev;
    TAG *ptr;
    
    object = space_info[space].list;
    prev = NULL;
    
    while(object != NULL) {
	/* If we are removing this object */
	if (Removed(object)) {
	    
	    /* Delete any pending timers */
	    del_timer(object, NULL);
	    
	    /* Remove ships from contact lists of other ships in
	       its space */
	    for (ptr = space_info[space].list; ptr != NULL; ptr=ptr->next)
		if (ptr != object)
		    snsRemoveContact(ptr, object, 0);
	    
	    /* Remove ship's own contacts */
	    while (object->contact_list != NULL)
		snsRemoveContact(object, object->contact_list->listref, 1);
	    
	    /* remove object from space list */
	    if (prev)
		prev->next = object->next;
	    else
		space_info[space].list = object->next;
	    
	    if (Ship(object)) {
		damWriteDamage(object->shipdata);
		
#ifdef ENABLE_ODOMETER
		navWriteOdometer(object->rec->db_obj, object->shipdata);
#endif
		
#ifdef ENABLE_TIME_TRACKER
		navWriteTimeActive(object->rec->db_obj, object->shipdata);
#endif
		
	    }
	    
	    if (space_info[space].flags & SPACE_LOGGED)
		log_space("Removed %s (#%d) from real space.", 
			  object->rec->name, object->rec->db_obj); 
	    
	    ptr = object->next;
	    
	    snsDelObject(object, space);
	    clsRemoveRecord(object->rec);

	    object = ptr;
	}
	else {
	    prev = object;
	    object = object->next;
	}
    }
    
    return;
}

int shipValidate(Record *recptr)
{
    int invalid = 0;
    int nconsoles = 0;
    unsigned int flags;

    if (F_FLOAT(recptr, turn_factor) <= 0.0) {
	log_space("#%d: Invalid turn factor (%3.2f). Must be greater than 0."
		  " Default (1.0) used.", recptr->db_obj, 
		  F_FLOAT(recptr, turn_factor));
	F_FLOAT(recptr, turn_factor) = 1.0;
    }
    
    if (F_FLOAT(recptr, roll_factor) <= 0.0) {
	log_space("#%d: Invalid roll factor (%3.2f). Must be greater than "
		  "0. Default (1.0) used.", recptr->db_obj, 
		  F_FLOAT(recptr, roll_factor));
	F_FLOAT(recptr, roll_factor) = 1.0;
    }
    
    if ((F_INT(recptr, shield_config) < 0) ||
	(F_INT(recptr, shield_config) >= NUM_SHIELD_CONFIGS)) {

	log_space("#%d:  Invalid shield config. Must be between 0 and %d.", 
		  recptr->db_obj, NUM_SHIELD_CONFIGS-1);
	return(1);
	
    }
    
    if (clsValidateDbref(F_PTR(recptr, dbref_bridge))) {
	log_space("#%d: Invalid bridge dbref (#%d). Object not loaded.",
		  recptr->db_obj, F_PTR(recptr, dbref_bridge));
	return(1);
    }
    if (clsValidateDbref(F_PTR(recptr, dbref_eng))) {
	log_space("#%d: Invalid engineering dbref (#%d). Object not loaded.",
		  recptr->db_obj, F_PTR(recptr, dbref_eng));
	return(1);
    }

    if (clsValidateList(F_PTR(recptr, consoles), clsValidateDbref)) {
	log_space("#%d: Invalid console dbref in list.", recptr->db_obj);
	return(1);
    }
    if (clsValidateList(F_PTR(recptr, consoles), validate_console))
	invalid++;
    else
	nconsoles++;

    if (nconsoles == 0) {
	log_space("#%d: No (valid) consoles were present. Object not loaded.", 
		  recptr->db_obj);
	return(1);
    }
    
    if (invalid) {
	log_space("#%d: Invalid console(s) found. Object not loaded.",  
		  recptr->db_obj);
	return(1);
    }
    
    /* initialize damage registers */
    
    if (strlen(F_PTR(recptr, system_status)) != NUM_SYSTEMS) {
	
	log_space("#%d: System damage string invalid. "
		  "Actual/Expected Length %d/%d.",
		  recptr->db_obj, strlen(F_PTR(recptr, system_status)),
		  NUM_SYSTEMS);
	return(1);
    }

    /* Validate/convert ship flags */
    if (loadSymbolicFlags(NULL, FLIST_SHIP, F_PTR(recptr, shipflags)) != NULL) {	
	log_space("#%d: Non-symbolic ship flags found. Trying old-style.",
		  recptr->db_obj);
	if (loadBinaryFlags(&flags, F_PTR(recptr, shipflags)) == 0) {
	    log_space("#%d: Not binary flags either. Giving up.", recptr->db_obj);
	    return 1;
	}
	
	log_space("#%d: Non-symbolic ship flags updated [%s].",
		  recptr->db_obj, F_PTR(recptr, shipflags));
	clsSetString(recptr, shipflags, saveSymbolicFlags(flags, FLIST_SHIP, 0));
	F_SET_WRITE_BACK(recptr, shipflags);
    }

    if (F_INT(recptr, door_present)) {
	if (F_INT(recptr, door_side) < 0 || F_INT(recptr, door_side) >= NUM_SHIELDS) {
	    log_space("#%d: Door side must be between 0 and %d.",
		      recptr->db_obj, NUM_SHIELDS-1);
	    return 1;
	}
    }

    if (F_FLOAT(recptr, warp_accel) == 0.0)
	log_space("#%d: Warp acceleration is 0.0. Ship cannot move!", 
		  recptr->db_obj);
    
    if (F_INT(recptr, gun_qty) > MAX_GUNS) {
	log_space("#%d: Too many guns specified. Using %d.", recptr->db_obj, 
		  MAX_GUNS);
	
	F_INT(recptr, gun_qty) = MAX_GUNS;
    }

    if (F_INT(recptr, torp_qty) > MAX_TORPS) {
	log_space("#%d: Too many torps specified. Using %d.", recptr->db_obj, 
		  MAX_TORPS);

	F_INT(recptr, torp_qty) = MAX_TORPS;
    }

    if (F_EXISTS(recptr, tractor_present) && F_INT(recptr, tractor_present)) {
	if (!F_EXISTS(recptr, tractor_effect)) {
	    log_space("#%d: Tractor is present, but CF_TRACTOR_EFFECT is not specified.", recptr->db_obj);
	    return 1;
	}
    }

    if (F_INT(recptr, damcon_teams) > MAX_TEAMS) {
	log_space("#%d: Too many damage control teams specified. Using %d.",
		  recptr->db_obj, MAX_TEAMS);
	
	F_INT(recptr, damcon_teams) = MAX_TEAMS;
    }

    if (F_INT(recptr, max_overload_points) == 0) {
	log_space("#%d:  Bad max_overload_points (0).", recptr->db_obj);
	return 1;
    }

    if (F_FLOAT(recptr, warp_factor) == 0.0) {
	log_space("#%d:  Bad warp_factor (0.0).", recptr->db_obj);
	return 1;
    }

    if (F_INT(recptr, hull) <= 0) {
	log_space("#%d:  Bad hull_integrity (%d)", recptr->db_obj, 
		  F_INT(recptr, hull));
	return 1;
    }

    /* Everything looks good! */
    return 0;

}

int validate_console(const char *console)
{
    dbref cdbref;
    char *flags;
    const char *badtype;
    
    cdbref = parseDbref(console);

    flags = pse_malloc(MAX_ATTRIBUTE_LEN);
    getAttrByNameLen(flags, cdbref, CONFIG_CONSOLE_TYPES, MAX_ATTRIBUTE_LEN);
    
    if (*flags == '\0') {
	log_space("Invalid console (#%s).", console);
	pse_free(flags);
	return(1);
    }
    
    badtype = loadSymbolicFlags(NULL, FLIST_CONSOLE, flags);
    
    if (badtype != NULL) {
	log_space("Invalid console type (#%d:%s).", cdbref, badtype);
	pse_free(flags);
	return(1);
    }
    
    pse_free(flags);
    return(0);
}

void shipAdd(Record *recptr)
{
    SHIP *ship;
    char *buff;
    const char *badflag;
    int nconsoles;
    const char *cn;
    int i;
    dbref consoledb;
    int t1, t2;
    
    ship = pse_malloc(sizeof(SHIP));
    ship->rec = recptr;
    recptr->ref = (void *) ship;
    
#ifdef ENABLE_TIME_TRACKER
    ship->time_start = time(NULL);
#endif
    
    cn = F_PTR(recptr, consoles);
    nconsoles = 0;
    
    buff = pse_malloc(MAX_ATTRIBUTE_LEN);
    
    while((NULL != (cn = next_dbref(cn, &consoledb))) &&
	  (nconsoles < MAX_CONSOLES)) {
	
	ship->consoles[nconsoles].dbref = consoledb;
	
	getAttrByNameLen(buff, consoledb, CONFIG_CONSOLE_TYPES,
			 MAX_ATTRIBUTE_LEN);
	
	if ((*buff != '\0') &&
	    loadSymbolicFlags(&ship->consoles[nconsoles].flags,
			      FLIST_CONSOLE, buff) == NULL) 
	    nconsoles++;
    }
    
    ship->num_consoles = nconsoles;
    
    for (i = 0; i < NUM_SYSTEMS; i++) {
	if (buff[i] == 'X')
	    buff[i] = 'x';
	
	if (((buff[i] < '0') || (buff[i] > '9'))
	    && (buff[i] != 'x'))
	    ship->damage[i].status = '0';
	else
	    ship->damage[i].status = buff[i];
	
	ship->damage[i].teams = 0;
	ship->damage[i].maxteams = 2;
	ship->damage[i].time_to_fix = 0;
	
    }
    
    ship->damage[SYSTEM_HULL].maxteams = 4;
    
    /* Load ship flags */
    badflag = loadSymbolicFlags(&(ship->shipflags), FLIST_SHIP, 
				F_PTR(recptr, shipflags));
    assert(badflag == NULL);
    
    ship->dbref_bridge = parseDbref(F_PTR(recptr, dbref_bridge));
    ship->dbref_eng = parseDbref(F_PTR(recptr, dbref_eng));
    
    /* Docking bay doors */
    if (F_EXISTS(recptr, door_present) && F_INT(recptr, door_present))
	ship->door_status = DOORS_CLOSED;
    else
	ship->door_status = DOORS_NONE;
    
    /* Cloaking device */
    if (F_EXISTS(recptr, cloak_present) && F_INT(recptr, cloak_present))
	ship->cloak_status = CLOAK_OFF;
    else
	ship->cloak_status = CLOAK_NONE;
    
    ship->time_to_cloak = 0;
    
#ifndef ENABLE_SHIELD_CLASSES
    
    switch(F_INT(recptr, shield_config)) {
    case SHIELD_CONFIG_ORIG4:
    case SHIELD_CONFIG_ORANGE1:
    case SHIELD_CONFIG_ORANGE2:
	ship->shield_number = 4;
	break;
	
    case SHIELD_CONFIG_ORIG6:
    case SHIELD_CONFIG_CUBE:
	ship->shield_number = 6;
	break;
	
    case SHIELD_CONFIG_NONE:
        ship->shield_number = 0;
        break;
	
    case SHIELD_CONFIG_ONE:
        ship->shield_number = 1;
        break;
    }
    
    for (i = 0; i < NUM_SHIELDS; i++) {
	ship->shield_max[i] = F_INT(recptr, shield_power);
	
	ship->shield_max_charge[i] = F_INT(recptr, shield_max_charge);
    }
#else
    /* shield class initialization */
    
    /* Clean out the layer settings */
    for (i = 0; i < SCM_MAX_LAYERS; i++) {
	ship->sc[i].name[0]  = '\0';
	ship->sc[i].plugin   = NULL;
	ship->sc[i].storage  = NULL;
	ship->sc[i].geometry = NULL;
	ship->sc[i].attrib   = -1;
    }
    
    ship->sc_num = 0;
    
    /* Attempt to load in the next shield layer */
    for (i = 0; i < SCM_MAX_LAYERS; i++) {
	
	sprintf(buff, "%s%d", SHIELDARRAY_LAYER, i);
	
	/* Grab the SCM stuff directly from the object for now. */
        getAttrByNameLen(buff, recptr->db_obj, buff, MAX_ATTRIBUTE_LEN);
	
	log_space("SCM: #%d: %s", recptr->db_obj, buff);
	
	if (scmLoadData(recptr->db_obj, &ship->sc[ship->sc_num], buff)) {

	    ship->sc[ship->sc_num].attrib = i;
	    ship->sc_num++;

	}
    }
#endif

    pse_free(buff);

    for (i = 0; i < F_INT(recptr, gun_qty); i++) {
	ship->gun[i].status = GUN_OFFLINE;
	ship->gun[i].power = F_INT(recptr, gun_power);
	ship->gun[i].range = F_INT(recptr, gun_range);
	
	ship->gun[i].range_percent = F_FLOAT(recptr, gun_range_percent);
	
	ship->gun[i].charge_per_turn = F_INT(recptr, gun_charge_rate);

	ship->gun[i].arc = F_INT(recptr, gun_arc); 
    }

    if (!F_EXISTS(recptr, torp_ammo))
	F_INT(recptr, torp_ammo) = -1;

    for (i = 0; i < F_INT(recptr, torp_qty); i++) {
	ship->torp[i].power = F_INT(recptr, torp_power);
	ship->torp[i].range = F_INT(recptr, torp_range);

	ship->torp[i].charge_per_turn = F_INT(recptr, torp_charge_rate);
	ship->torp[i].base_accuracy = F_FLOAT(recptr, torp_accuracy);
    }

    if (F_EXISTS(recptr, torp_loaders))
	ship->free_torp_loaders = F_INT(recptr, torp_loaders);
    else
	ship->free_torp_loaders = 0;

    if (F_EXISTS(recptr, tractor_present) && F_INT(recptr, tractor_present))
	ship->tractor_status = TRACTOR_OFF;
    else
	ship->tractor_status = TRACTOR_NONE;

    sscanf(F_PTR(recptr, hull_status), "%d %d", 
	   &t1, &t2);
    ship->current_integrity = F_INT(recptr, hull) - t1;
    ship->permanent_hull_damage = t2;

    if (ship->current_integrity < 0)
	ship->shipflags |= DISABLED;
    
    ship->reactor_setting = 5;
    ship->reactor_drain = 0;
    ship->overload_points = 0;
    ship->eng_warnings = 1; /* set warnings on */
    ship->eng_safeties = 1; /* set automatic overrides on */
    ship->battery_discharge = 0;
    ship->battery_level = F_INT(recptr, battery_capacity);
    
    /* cloak */
    if (ship->cloak_status == CLOAK_NONE)
	ship->damage[SYSTEM_CLOAK].status = 'x';
    
    /* tractor */
    if (ship->tractor_status == TRACTOR_NONE)
	ship->damage[SYSTEM_TRACTOR].status = 'x';
    
    /* default allocations */
    ship->alloc_nav = 0;
    ship->alloc_batt = 0;
    ship->alloc_tac = 0;
    ship->alloc_nav = 0;
    ship->alloc_shields = 0;
    ship->talloc_guns = 0;
    ship->talloc_torps = 0;
    ship->talloc_tractor = 0;
    ship->tractor_target = NULL;
    ship->tractor_power = 0;
    ship->num_guns_online = 0;
    ship->num_torps_online = 0;
    
    ship->warp_speed = 0.0;
    ship->warp_set_speed = 0.0;
    
    /* shield stuff */
    for (i = 0; i < NUM_SHIELDS; i++) {
	ship->shield_level[i] = 0;
	ship->shield_alloc[i] = 0;
	ship->shield_status[i] = SHLD_READY;
	ship->shield_action[i] = SHLD_STABLE;
    }
    
    /* gun stuff */
    for (i = 0; i < F_INT(recptr, gun_qty); i++) {
	ship->gun[i].status = GUN_OFFLINE;
	ship->gun[i].charge = 0;
    }
    
    for (i = F_INT(recptr, gun_qty); i < MAX_GUNS; i++)
	ship->damage[SYSTEM_GUN_0 + i].status = 'x';
    
    /* torp stuff */
    ship->auto_online = FALSE;
    ship->auto_reload = TRUE;
    
    for (i = 0; i < F_INT(recptr, torp_qty); i++) {
	ship->torp[i].status = TORP_LOADED;
	ship->torp[i].charge = 0;
	ship->torp[i].turns_charged = 0;
    }
    
    for (i = F_INT(recptr, torp_qty); i < MAX_TORPS; i++)
	ship->damage[SYSTEM_TORP_0 + i].status = 'x';
    
    /* initialize comm configuration */
    for (i = 0; i < MAX_COMM_CHANNELS; i++) {
	ship->channel[i].frequency = 0;
	ship->channel[i].send_status = CS_COMM_ONLY;
	ship->channel[i].receive_status = CR_COMM_ONLY;
	strcpy(ship->channel[i].label, "");
    }
    
    /* initialize damage control team status */
    for (i = 0; i < F_INT(recptr, damcon_teams); i++)
	ship->team[i] = -1;

#ifdef ENABLE_ODOMETER
    if (!F_EXISTS(recptr, odometer)) {
	F_INT(recptr, odometer) = 0;
	F_SET_WRITE_BACK(recptr, odometer);
    }
#endif

#ifdef ENABLE_TIME_TRACKER
    if (!F_EXISTS(recptr, time_active)) {
	F_INT(recptr, time_active) = 0;
	F_SET_WRITE_BACK(recptr, time_active);
    }
#endif

}

/*
 * Build the output into buff before returning it.
 */
void objDisplaySpecs(TAG *object, void *mud, int godmode)
{
    SSTR *ps;
    char *buff;
    SHIP *ship = object->shipdata;
#ifdef ENABLE_SHIELD_CLASSES
    int i;
#endif
    
    buff = pse_malloc(MAX_ATTRIBUTE_LEN);
    ps = sstr_new(MAX_ATTRIBUTE_LEN);

    if (Ship(object))
	sprintf(buff, "%s -- %s %s-class %s\n", object->rec->name,
		F_PTR(ship->rec, owner_name), F_PTR(ship->rec, shipclass), 
		F_PTR(ship->rec, type));
    else
	sprintf(buff, "%s\n", object->rec->name);
    
    sstr_cat(ps, buff, 0);

    if (godmode) {
	sprintf(buff, "Flags: %s\nSize: %2.1f\n",
		saveSymbolicFlags(object->tagflags, FLIST_OBJECT, 0),
		F_FLOAT(object->rec, size));
	
	sstr_cat(ps, buff, 0);
    }
    
    if (CanSense(object)) {
	sprintf(buff, "Sensor Range: %.0f %s\n",
		F_INT(object->rec, sensor_range) * 
		F_FLOAT(object->rec, range_factor),
		F_PTR(object->rec, range_name));
	
	sstr_cat(ps, buff, 0);
    }
    
    if (F_INT(object->rec, xport_range) > 0) {
	sprintf(buff, "Transporter Range: %d\n", 
		F_INT(object->rec, xport_range));
	
	sstr_cat(ps, buff, 0);
    }
    
    if (EventDriven(object)) {
	sprintf(buff, "Critical Range: %.0f %s\n",
		F_INT(object->rec, critical_range) * 
		F_FLOAT(object->rec, range_factor),
		F_PTR(object->rec, range_name));
	
	sstr_cat(ps, buff, 0);
    }
    
    if (F_FLOAT(object->rec, cloak_effect) > 0) {
	sprintf(buff, "Cloak Effectiveness: %1.1f\n", 
		F_FLOAT(object->rec, cloak_effect));
	
	sstr_cat(ps, buff, 0);
    }
    
    if (Ship(object)) {
	
	if (godmode) {
	    sprintf(buff, "Ship Flags: %s\n", 
		    saveSymbolicFlags(ship->shipflags, FLIST_SHIP, 0));
	    
	    sstr_cat(ps, buff, 0);
	}
	
	sprintf(buff, "Engine output max: %d    Optimum running "
		"level: %d%%\n"
		"Battery capacity: %d    Discharge rate: %d\n"
		"Turn factor: %4.2f          Roll factor: %4.2f\n",
		F_INT(ship->rec, reactor_output_max),
		F_INT(ship->rec, reactor_stress_level),
		F_INT(ship->rec, battery_capacity),
		F_INT(ship->rec, battery_discharge_max),
		F_FLOAT(ship->rec, turn_factor),
		F_FLOAT(ship->rec, roll_factor));
	
	sstr_cat(ps, buff, 0);
	
	if (ship->door_status != DOORS_NONE)
	    sprintf(buff, "Hull integrity: %5d      Door size: %2.1f\n",
		    F_INT(ship->rec, hull), 
		    F_FLOAT(ship->rec, door_size));
	else
	    sprintf(buff, "Hull integrity: %5d\n", 
		    F_INT(ship->rec, hull));
	
	sstr_cat(ps, buff, 0);
	
	if (ship->cloak_status != CLOAK_NONE) {
	    sprintf(buff, "Cloaking Device: Yes       Cost: %d\n", 
		    F_INT(ship->rec, cloak_cost));
	    sstr_cat(ps, buff, 0);
	} else
	    sstr_cat(ps, "Cloaking Device: No\n", 0);
	
	if (ship->tractor_status != TRACTOR_NONE) {
	    sprintf(buff, "Tractor Beam: Yes          "
		    "Effectiveness: %3.1f\n", 
		    F_FLOAT(ship->rec, tractor_effect));
	    sstr_cat(ps, buff, 0);
	} else
	    sstr_cat(ps, "Tractor Beam: No\n", 0);
	
	sprintf(buff, "Scanner Range: %.0f %s\n",
		F_INT(ship->rec, scanner_range) * 
		F_FLOAT(object->rec, range_factor),
		F_PTR(object->rec, range_name));
	
	sstr_cat(ps, buff, 0);
	
#ifdef ENABLE_SHIELD_CLASSES
	if (ship->sc_num > 0) {
	    for (i = 0; i < ship->sc_num; i++) {

		sprintf(buff, "Layer%d: %s\n", i, ship->sc[i].name);

		sstr_cat(ps, buff, 0);
	    }
	} else {
	    sstr_cat(ps, "No shielding layers defined.\n", 0);
	}
#else
	sprintf(buff, "Shield Array: %s\n"
		"Maximum normal shield charge: %d\n",
		shdConfigName(object), 
		ship->shield_max[0]);
	
	sstr_cat(ps, buff, 0);	
#endif
	
	if (F_INT(ship->rec, gun_qty))
	    sprintf(buff, "%s banks: %d   Maximum charge: %d   "
		    "Effective range: %d\n",
		    capstr(F_PTR(ship->rec, gun_string)), 
		    F_INT(ship->rec, gun_qty),
		    ship->gun[0].power,
		    (int)(0.8 * (float)(ship->gun[0].range)));
	else
	    sprintf(buff, "%s banks: 0\n", 
		    capstr(F_PTR(ship->rec, gun_string)));
	
	sstr_cat(ps, buff, 0);	
	
	if (F_INT(ship->rec, torp_qty)) 
	    sprintf(buff, "%s tubes: %d   Arm: %d  "
		    "Power: %d  Effective range: %d  Reloaders: %d\n", 
		    capstr(F_PTR(ship->rec, torp_string)), 
		    F_INT(ship->rec, torp_qty),
		    ship->torp[0].power / 2, ship->torp[0].power,
		    (int)(0.6 * (float)(ship->torp[0].range)),
		    F_INT(ship->rec, torp_loaders));
	else
	    sprintf(buff, "%s tubes: 0\n", 
		    capstr(F_PTR(ship->rec, torp_string)));
	
	sstr_cat(ps, buff, 0);
    }
    
    WriteFunctionBuffer(sstr_str(ps));
    
    pse_free(buff);
    sstr_free(ps);
    
    return;
}

/*
 * Return either a preformatted screen, or a list of dbrefs.
 */
void objList(void *mud, int space, int show_ships, int show_other, int dbs)
{
    XYZ pos;
    int count;
    SPH bearing;
    TAG *ptr;
    SSTR *ps;
    char *buff;
    struct timeval now;
    
    ps = sstr_new(MAX_ATTRIBUTE_LEN);
    buff = pse_malloc(MAX_ATTRIBUTE_LEN);
    
    gettimeofday(&now, NULL);

    if (dbs == 0) {
	sprintf(buff, "%s\n---------------------------------------------"
		"-------------------------------\n",space_info[space].name);
	sstr_cat(ps, buff, 0);
    }
    
    count = 0;
    
    for (ptr = space_info[space].list; ptr != NULL; ptr=ptr->next) {
	
	count++;
	if (!show_ships && Ship(ptr)) continue;
	if (!show_other && !Ship(ptr)) continue;
	
	if (dbs) {
	    sprintf(buff, "#%d ", ptr->rec->db_obj);
	    sstr_cat(ps, buff, 0);
	} else {
	    
	    object_position(ptr, &pos, &now, 0);
	    
	    xyz_to_sph(pos, &bearing);
	    
	    if (!Ship(ptr) && ptr->speed==0) 
		sprintf(buff,"#%-5d %-30s  %3.0f%-+4.0f " RANGEF_NP
			"\t\t%c%c\n", 
			ptr->rec->db_obj, ptr->rec->name,
			bearing.bearing, bearing.elevation, bearing.range,
			(ptr->locked_on != NULL) ? 'L' : ' ',
			Cloaked(ptr) ? 'C' : ' ');
	    else
		sprintf(buff, "#%-5d %-30s  %3.0f%-+4.0f " RANGEF_NP
			"  %3.0f%-+4.0f %4.1f\t%c%c\n",
			ptr->rec->db_obj, ptr->rec->name, 
			bearing.bearing, bearing.elevation, 
			bearing.range, ptr->heading.bearing, 
			ptr->heading.elevation, ptr->speed,
			(ptr->locked_on != NULL) ? 'L' : ' ',
			Cloaked(ptr) ? 'C' : ' ');
	    
	    sstr_cat(ps, buff, 0);
	}
    }
    
    if (dbs == 0) {
	sprintf(buff, "---------------------------------------------"
		"-------------------------------\n"
		"Total objects:  %d", count);
	sstr_cat(ps, buff, 0);
    }
    
    WriteFunctionBuffer(sstr_str(ps));
    
    sstr_free(ps);
    pse_free(buff);
    
    return;
}

TAG *objFindObject(dbref target)
{
     int i;
     TAG *object;

     for (i = 0; i < NUM_SPACES; i++) { 
	  for (object = space_info[i].list; object != NULL; 
	       object=object->next)
	       
	       if (object->rec->db_obj == target)
		    return(object);
     }

     return (NULL);

} 

/*
 * Matches the provided name with the system identification
 * strings.
 * 
 * Returns: -1 Ambiguous
 *	    -2 Unknown
 *	    0-NUM_SYSTEMS Matched System ID
 *
 */
int objSystemNameToInt(const TAG *object, char *system)
{
    SHIP *ship = object->shipdata;
    int found = 0;
    int sys;
    int i;
    unsigned int len;
    char *str;
    
    len = strlen(system);
    
    if ((len == 0) || (!Ship(object)))
	return -2;
    
    str = pse_malloc(MAX_ATTRIBUTE_LEN);
    
    strncpy(str, system, MAX_ATTRIBUTE_LEN);
    str[MAX_ATTRIBUTE_LEN-1] = '\0';
    
    str[0] = toupper(str[0]);
    for (i = 1; str[i]; i++) {
	if (str[i-1] == '-')
	    str[i] = toupper(str[i]);
	else
	    str[i] = tolower(str[i]);
    }
    
#ifdef ENABLE_SHIELD_CLASSES
    /** FIXME */
    sys = 0;
#else
    for (i=sys=0; i < ship->shield_number; i++)
	if (strncmp(str, shdFullName(object,i), len) == 0) {
	    sys = i;
	    found++;
	}
#endif
    
    str[0] = tolower(str[0]);
    
    /* figure out which system we're fixing */
    for (i=NUM_SHIELDS; i < NUM_SYSTEMS; i++) 
	if (strncmp(str, system_names[i], len) == 0) {
	    sys = i;
	    found++;
	}
    
    pse_free(str);
    
    if (!found)
	/* we didn't find the system.*/
	return -2;
    
    if (found != 1)
	/* Found more than 1 */
	return -1;
    
    return sys;
}

const char* objSystemIntToName(const TAG *object, int system)
{
    if (!Ship(object))
	return "";
    
    if (system < NUM_SHIELDS)
	return shdFullName(object,system);
    
    if (system >= NUM_SYSTEMS)
	return "";
    
    return system_names[system];
}

void objSetObjectFlag(TAG *object, void *mud, int value, char *name)
{
    if (setFlag(&(object->tagflags), value, FLIST_OBJECT, name))
	FWriteFunctionBuffer("#-1 Invalid flag (%s).", name);
    else {
	if (strcmp(name, "SHIP") == 0) {
	    WriteFunctionBuffer("#-1 SHIP FLAG CANNOT BE CHANGED");
	    return;
	}
	
	if (value)
	    WriteFunctionBuffer("Set.");
	else
	    WriteFunctionBuffer("Cleared.");
	
	if (space_info[object->space].flags & SPACE_LOGGED) {
	    
	    log_space("Flag changed on %s (#%d) %s%s.",
		      object->rec->name, object->rec->db_obj, 
		      value ? "" : "!", name);
	}
    }
    
    return;
}

void objSetShipFlag(TAG *object, void *mud, int value, char *name)
{
    if (setFlag(&(object->shipdata->shipflags), value, FLIST_SHIP,
		name))
	FWriteFunctionBuffer("#-1 Invalid flag (%s).", name);
    else {
	if (value)
	    WriteFunctionBuffer("Set.");
	else
	    WriteFunctionBuffer("Cleared.");
	
	if (space_info[object->space].flags & SPACE_LOGGED) {
	    
	    log_space("Ship flag changed on %s (#%d) %s%s.",
		      object->rec->name, object->rec->db_obj, 
		      value ? "" : "!", name);
	}
    }
    
    return;
}

int objHasObjectFlag(const TAG *object, dbref player, const char *name)
{
    return (isFlagSet(object->tagflags, FLIST_OBJECT,  name));
}

int objHasShipFlag(const TAG *object, dbref player, const char *name)
{
    return (isFlagSet(object->shipdata->shipflags, FLIST_SHIP, name));
}

int objHasConsoleFlag(const TAG *object, dbref player, int consoleno,
		      const char *name)
{
    return (isFlagSet(object->shipdata->consoles[consoleno].flags,
		       FLIST_CONSOLE, name));
}
