/* v1.1
 *
 * output.c:  Output formatting code.
 *
 * When building up output code, we often need to join strings, safely
 * append, etc. This module contains a set of routines to permit us to
 * do this. Essentially it's a string class. Each 'string' is prefixed
 * with a structure that contains information about the string - it's
 * maximum length, current length, etc.
 *
 * This program is free software and may be freely redistributed as
 * specified in the GNU General Public License.  Please see the file
 * 'COPYING' for details.
 */

#include <assert.h>
#include <stdlib.h>
#include <ctype.h>
#include <stdarg.h>

#include "spaceconf.h"
#include "pseint.h"
#include "dbint.h"
#include "space.h"
#include "output.h"

/*
 * Allocate a new (empty) string of the given length.
 */
SSTR* sstr_new(size_t length)
{
    SSTR *ps;
    
    assert(length != 0);
    
#ifdef DEBUG_SSTR
    ps = (SSTR*) pse_malloc(sizeof(SSTR) + sizeof(char) * (length + 4));
    ps->trap = 0x12345678;
    ps->str[length+0] = '1';
    ps->str[length+1] = '2';
    ps->str[length+2] = '3';
    ps->str[length+3] = '4';
#else
    ps = (SSTR*) pse_malloc(sizeof(SSTR) + sizeof(char) * (length));
#endif
    
    ps->max = length;
    ps->cur = 0;
    ps->str[0] = '\0';
    return ps;
}

#ifdef DEBUG_SSTR
void sstr_assert(SSTR* ps)
{
    assert(ps != NULL);
    assert(ps->trap == 0x12345678);
    assert(ps->str[length+0] == '1');
    assert(ps->str[length+1] == '2');
    assert(ps->str[length+2] == '3');
    assert(ps->str[length+3] == '4');
    assert(ps->cur == strlen(ps->str));
}
#endif

/*
 * Free the string
 */
void sstr_free(SSTR *ps)
{
#ifdef DEBUG_SSTR
    sstr_assert(ps);
#endif
    
    pse_free(ps);
}

/*
 * If length is given as 0, we work it out here.
 *
 * 0: Truncated.
 * 1: Appended.
 */
int sstr_cat(SSTR *ps, const char *str, size_t len)
{
#ifdef DEBUG_SSTR
    sstr_assert(ps);
#endif
    
    /* If string is already full, give up */
    if (ps->cur == ps->max)
	return 0;
    
    /* If no string to append, return */
    if ((str == NULL) || (*str == '\0'))
	return 1;
    
    if (len == 0)
	len = strlen(str);
    
    if (len == 0)
	return 1;
    
    /* If we have to truncate, then truncate, else just do a copy. */
    if (len + ps->cur > ps->max - 1) {
	strncpy(&ps->str[ps->cur], str, ps->max - ps->cur - 1);
	ps->cur = ps->max;
	ps->str[ps->max - 1] = '\0';
	return 0;
    } else {
	strncpy(&ps->str[ps->cur], str, len);
	ps->cur += len;
	ps->str[ps->cur] = '\0';
	return 1;
    }
}

/*
 * Note: we play it safe, and don't assume anything about the return
 *       from vsprintf. It varies greatly between platforms last I
 *       checked. We should also use vsnprintf if we could
 *       guarantee it exists on every platform.
 *
 * Note: We use a large static buffer. Ugly, but we need some temporary
 *       storage to sprintf into, and this function can never be reentered
 *       (unless someone makes us multi-threaded)
 *
 * 0: Truncated.
 * 1: Appended.
 */
int sstr_cat_sprintf(SSTR *ps, const char *fmt, ...)
{
    va_list list;
    static char buff[LARGE_BUF_SIZE];
    
#ifdef DEBUG_SSTR
    sstr_assert(ps);
#endif
    
    /* If string is already full, return immediate truncation */
    if (ps->cur == ps->max)
	return 0;
    
    /* If no string to append, return immediate success */
    if ((fmt == NULL) || (*fmt == '\0'))
	return 1;
    
    va_start(list, fmt);
    vsprintf(buff, fmt, list);
    va_end(list);

    return sstr_cat(ps, buff, 0);
}

/*
 * Note: we play it safe, and don't assume anything about the return
 *       from vsprintf. It varies greatly between platforms last I
 *       checked. We should also use vsnprintf if we could
 *       guarantee it exists on every platform.
 *
 * Note: We use a large static buffer. Ugly, but we need some temporary
 *       storage to sprintf into, and this function can never be reentered
 *       (unless someone makes us multi-threaded)
 *
 * 0: Truncated.
 * 1: Appended.
 */
int sstr_sprintf(SSTR *ps, const char *fmt, ...)
{
    va_list list;
    static char buff[LARGE_BUF_SIZE];
    
#ifdef DEBUG_SSTR
    sstr_assert(ps);
#endif

    /* Zap the current contents of the safe string */
    ps->cur = 0;
    ps->str[0] = '\0';

    /* If no string to append, return immediate success */
    if ((fmt == NULL) || (*fmt == '\0'))
	return 1;
    
    va_start(list, fmt);
    vsprintf(buff, fmt, list);
    va_end(list);

    return sstr_cat(ps, buff, 0);
}

/*
 * Provides access to the string as a character array.
 */
const char *sstr_str(const SSTR *ps)
{
#ifdef DEBUG_SSTR
    sstr_assert(ps);
#endif
    
    return ps->str;
}

/*
 * Provides access to the length of the string.
 */
size_t sstr_len(const SSTR *ps)
{
#ifdef DEBUG_SSTR
    sstr_assert(ps);
#endif

    return ps->cur;
}
