/* v1.1
 *
 * platform.c:  Platform-specific code.
 *
 * This program is free software and may be freely redistributed as
 * specified in the GNU General Public License.  Please see the file
 * 'COPYING' for details.
 */

#include <assert.h>
#include <stdlib.h>
#include <stdarg.h>
#include <ctype.h>
#include "platform.h"
#include "spaceconf.h"
#include "pseint.h"
#include "space.h"

/* Currently for Penn, TinyMux and TinyMush, the default is GOD */
dbref debug_char=GOD;

/* Note: some bits of PENN are sufficiently different that it's handled
 *       separately later.
 */
#if defined(MUX) || defined(MUSH) || defined(MUSH3) || defined(MUX2)

/* 
 * evalString:  Evaluate a string using the platform's 'exec' function.
 *
 * The string to be evaluated is provided in *str, and parameters in
 * cargs/ncargs.
 */
void evalString(char *buff, dbref obj, char *str, char **cargs, 
		int ncargs)
{
    char *tmpbuf;

#if defined(MUX2)
    char *bp;

    tmpbuf = alloc_lbuf("getEvalAttr");

    bp = tmpbuf;
    TinyExec(tmpbuf, &bp, 0, obj, GOD, EV_EVAL | EV_FIGNORE | EV_TOP,
	 &str, cargs, ncargs);
    *bp='\0';
#else
#if defined(MUX) || defined(MUSH3)
    char *bp;

    tmpbuf = alloc_lbuf("getEvalAttr");

    bp = tmpbuf;
    exec(tmpbuf, &bp, 0, obj, GOD, EV_EVAL | EV_FIGNORE | EV_TOP,
	 &str, cargs, ncargs);
    *bp='\0';
#else
#if defined(MUSH)
    tmpbuf = exec(obj, GOD, EV_EVAL | EV_FIGNORE | EV_TOP,
		  str, cargs, ncargs);
#else
#error No platform defined.
#endif
#endif
#endif

    strncpy(buff, tmpbuf, MAX_ATTRIBUTE_LEN);
    buff[MAX_ATTRIBUTE_LEN - 1] = '\0';

    /*
     * We must free the buffer we allocated (MUX), or that was allocated
     * for us (MUSH).
     */
    free_lbuf(tmpbuf);

    return;
}

#if defined(MUSH3)
/*
 * getAttrByNumberLen:  Fetch an attribute string into the buffer provided.
 */
void getAttrByNumberLen(char *buff, dbref obj, int attrnum, int max)
{
    dbref aowner;
    int aflags;
    char *atr_gotten;
    int alen;

    /* Check for the special attribute cases */
    switch(attrnum) {
    case PSE_DATA_DBREF:
	attrnum = A_DATA_DBREF;
	break;
	
    case PSE_USER_DBREF:
	attrnum = A_USER_DBREF;
	break;
    }
    
    /* Retrieve the attribute */
    atr_gotten = atr_pget(obj, attrnum, &aowner, &aflags, &alen);

    if (!atr_gotten) {
	buff[0] = '\0';
	return;
    }
    
    /* Optimize the copy as TinyMush 3 provide the length of the string. 
     * We need +1 as we want the terminator too!
     */
    if ((alen > 0) && (alen < max))
	max = alen+1;
    
    /* Copy attribute to target buffer and ensure it's terminated */
    strncpy(buff, atr_gotten, max);
    buff[max - 1] = '\0';

    /* Free the attribute buffer */
    free_lbuf(atr_gotten);
}

/*
 * getAttrByNameLen:  Fetch an attribute string into the buffer provided.
 */
void getAttrByNameLen(char *buff, dbref obj, const char *attrname, int max)
{
    ATTR *attr;

    /* Return the attribute information for the named attribute */
    attr = atr_str(attrname);

    /* If it exists, do the attribute fetch based on the number */
    if (attr)
	getAttrByNumberLen(buff, obj, attr->number, max);
    else
	buff[0] = '\0';
}

#else

/*
 * getAttr:  Fetch an attribute string into the buffer provided.
 */
void getAttrByNumberLen(char *buff, dbref obj, int attrnum, int max)
{
    dbref aowner;
    int aflags;
    char *atr_gotten;

    /* Check for the special attribute cases */
    switch(attrnum) {
    case PSE_DATA_DBREF:
	attrnum = A_DATA_DBREF;
	break;
	
    case PSE_USER_DBREF:
	attrnum = A_USER_DBREF;
	break;
    }
    
    /* Retrieve the attribute */
    atr_gotten = atr_pget(obj, attrnum, &aowner, &aflags);

    /* Copy attribute to target buffer and ensure it's terminated */
    strncpy(buff, atr_gotten, max);
    buff[max - 1] = '\0';

    /* Free the attribute buffer */
    free_lbuf(atr_gotten);
}

/*
 * getAttr:  Fetch an attribute string into the buffer provided.
 */
void getAttrByNameLen(char *buff, dbref obj, const char *attrname, int max)
{
    ATTR *attr;

#if defined(MUSH)
    /* Mush modifies the provided attribute name under some circumstances,
     * so we have to take a copy. Note that max should be long enough to
     * fit the attribute name. At the moment, max is always large enough.
     * If at some point this condition doesn't hold, we just have to add
     * an allocation and free call here.
     */
    strncpy(buff, attrname, max);
    buff[max-1] = '\0';

    attr = atr_str(buff);
#else
#if defined(MUX) || defined(MUX2)
    attr = atr_str(attrname);
#else
#error No platform defined.
#endif
#endif

    /* Lookup the attribute by name. Return an empty string if not found */
    if (attr == NULL)
	buff[0] = '\0';
    else
	getAttrByNumberLen(buff, obj, attr->number, max);

    return;
}
#endif

/*
 * getEvalAttrBuf:  Fetch an attribute string, and evaluate it.
 *                  Return the attribute in the buffer provided.
 *                  The provided buffer is set to a zero length
 *                  string if there is an error or the attribute
 *                  does not exist.
 */

void getEvalAttrBuf(dbref obj, const char *attrname, char **cargs,
		    unsigned int ncargs, char *buff)
{
    char *atbuff;
    ATTR *attr;

#if defined(MUSH) || defined(MUSH3)
    /*
     * Mush modifies the provided buffer, so we have to take a copy.
     */
    strncpy(buff, attrname, MAX_ATTRIBUTE_LEN);
    buff[MAX_ATTRIBUTE_LEN-1]='\0';
    
    attr = atr_str(buff);
#else
#if defined(MUX) || defined(MUX2)
    attr = atr_str(attrname);
#else
#error No platform defined.
#endif
#endif

    if (attr == NULL)
	/* Clear the provided buffer if the attribute id doesn't exist */
	buff[0] = '\0';
    else {
	
	/* Allocate memory and retrieve the attribute */
	atbuff = (char *) pse_malloc(MAX_ATTRIBUTE_LEN);
	getAttrByNumberLen(atbuff, obj, attr->number, MAX_ATTRIBUTE_LEN);
	
	/* Evaluate if the retrieved attribute wasn't empty */
	if (*atbuff != '\0')
	    evalString(buff, obj, atbuff, cargs, ncargs);
	else
	    buff[0] = '\0';
	
	pse_free(atbuff);
    }
    
    return;
}

/*
 * setAttr:  Record an attribute in the database. 
 */
void setAttrByNumber(dbref obj, int attrnum, const char *value)
{
     dbref aowner;
     int aflags;
     char *buff;

     if (ValidObject(obj)) {
	 
	 buff = (char *) pse_malloc(MAX_ATTRIBUTE_LEN);
	 
	 strncpy(buff, value, MAX_ATTRIBUTE_LEN);
	 buff[MAX_ATTRIBUTE_LEN - 1] = '\0';
	 atr_pget_info(obj, attrnum, &aowner, &aflags);
	 atr_add(obj, attrnum, buff, aowner, aflags);
	 
	 pse_free(buff);
     }
     
     return;
}

void setAttrByName(dbref obj, const char *attrname, const char *value)
{
     ATTR *attr;

#if defined(MUSH) || defined(MUSH3)
     /*
      * Mush modifies the provided buffer in some circumstances, so we have
      * to take a copy.
      */
     char *buff;

     buff = pse_malloc(MAX_ATTRIBUTE_LEN);

     strncpy(buff, attrname, MAX_ATTRIBUTE_LEN);
     buff[MAX_ATTRIBUTE_LEN-1]='\0';

     if ((attr=atr_str(buff)) != NULL)
	  setAttrByNumber(obj, attr->number, value);

     pse_free(buff);
#else
#if defined(MUX) || defined(MUX2)
     if ((attr=atr_str(attrname)) != NULL)
	  setAttrByNumber(obj, attr->number, value);
#else
#error No platform defined.
#endif
#endif

     return;
}

/* getAttrNumber: Takes an attribute name, and performs a lookup to return
 *                the corresponding number.
 *
 *                For Mush and Mux, we get a pointer to the ATTR struct,
 *                from which we can just use the number member.
 *
 *                Note: we must not free the ATTR struct!
 */

int getAttrNumber(const char *name)
{
     const ATTR *ap;

#if defined(MUSH) || defined(MUSH3)
     /*
      * Mush modifies the provided buffer in some circumstances, so we have
      * to take a copy.
      */
     char *buff;

     buff = pse_malloc(MAX_ATTRIBUTE_LEN);

     strncpy(buff, name, MAX_ATTRIBUTE_LEN);
     buff[MAX_ATTRIBUTE_LEN-1]='\0';

     ap = atr_str(buff);

     pse_free(buff);

     if (ap == NULL)
	  return 0;
#else
#if defined(MUX) || defined(MUX2)
     if (!(ap = atr_str(name)))
	  return 0;
#else
#error No platform defined.
#endif
#endif

     if (!(ap->number))
	  return -1;

     return ap->number;
}

#else
#if defined(PENN)

void getAttrByNumberLen(char *buff, dbref obj, int attrnum, int max)
{
     switch(attrnum)
     {
	case PSE_DATA_DBREF:
	    getAttrByNameLen(buff, obj, "DATADBREF", max);
	    return;

	case PSE_USER_DBREF:
	    getAttrByNameLen(buff, obj, "USERDBREF", max);
	    return;
     }

     buff[0] = '\0';
     return;
}

void getAttrByNameLen(char *buff, dbref obj, const char *attrname, int max)
{
     ATTR *attr;
     char *filter;

     buff[0] = '\0';

     if (!(attr=atr_get(obj, attrname)))
	  return;

     filter = safe_uncompress(attr->value);
     strncpy(buff, filter, max);
     buff[max - 1] = '\0';
     pse_free(filter);

     return;     
}

void getEvalAttrBuf(dbref obj, const char *attrname, char **cargs,
			unsigned int ncargs, char *rbuff)
{
    char *buff;
    ATTR *attr;
    char *temp;
    
    temp = buff = pse_malloc(MAX_ATTRIBUTE_LEN);

    buff[0] = '\0';
    
    if ((attr=atr_get(obj, attrname)) != NULL)
      	do_userfn(buff, &temp, obj, attr, ncargs, cargs, GOD, GOD, GOD, NULL);
    *temp = '\0';

    strncpy(rbuff, buff, MAX_ATTRIBUTE_LEN);
    pse_free(buff);

    rbuff[MAX_ATTRIBUTE_LEN-1] = '\0';
}

void setAttrByName(dbref obj, const char *attrname, const char *value)
{

     do_set_atr(obj, attrname, value, GOD, NOTHING);

}

int getAttrNumber(const char *attr)
{
	/* TODO: Fix this if/when we use varrays. We probably need to hook
                 a Penn call to return a unique hash of the attribute. */
	return 1;
}

#else
#error No platform defined.
#endif
#endif

/*
 * PSE version of code to strip ansi. We strip the code in-place, which
 * is safe as the code without ansi can't be longer than the original.
 * As we strip in place, we need to be sure that the buffer we provide
 * is temporary.
 */
void pse_strip_ansi(char *str)
{
    char *s, *t;
    
    /* Sanity check */
    if (str == NULL)
	return;
    
    s = t = str;
    
    while(*s) {
	if (*s == 0x1b) {
	    while(*s && !isalpha(*s))
		s++;
	    if (*s)
		s++;
	} else
	    *t++ = *s++;
    }
    
    *t = '\0';
}

/*
 * Platform indepenant interface functions to output responses.
 */

void Notify(dbref obj, const char *str)
{
    if (obj == PSE_NOTHING)
	return;
    
#if defined(MUX) || defined(MUSH) || defined(MUSH3) || defined(MUX2)
    notify_with_cause(obj, GOD, str);
#else
#if defined(PENN)
    notify_by(GOD, obj, str);
#else
#error No platform defined.
#endif
#endif
}

void NotifyExcept(dbref loc, dbref except, const char *str)
{
#if defined(MUX2)
    if (except == PSE_NOTHING)
	except = NOTHING;

    notify_except(loc, GOD, except, str, 0);
#else
#if defined(MUX) || defined(MUSH) || defined(MUSH3)
    if (except == PSE_NOTHING)
	except = NOTHING;

    notify_except(loc, GOD, except, str);
#else
#if defined(PENN)
    dbref ortemp;

    if (except == PSE_NOTHING)
	except = NOTHING;

    ortemp = orator;
    orator = GOD;
    notify_except(loc, except, str);
    orator = ortemp;
#else
#error No platform defined.
#endif
#endif
#endif
}

void NotifyLocation(dbref loc, const char *str)
{
#if defined(MUX2)
    if (loc == PSE_NOTHING)
	return;
    
    notify_except(loc, GOD, NOTHING, str, 0);
#else
#if defined(MUX) || defined(MUSH) || defined(MUSH3)
    if (loc == PSE_NOTHING)
	return;
    
    notify_except(loc, GOD, NOTHING, str);
#else
#if defined(PENN)
    dbref or = orator;

    if (loc == PSE_NOTHING)
	return;
    
    orator = GOD;
    notify_except(loc, NOTHING, str);
    orator = or;
#else
#error No platform defined.
#endif
#endif
#endif
}

/*
 * Platform independant functions used by the PSE to allocate and free buffers.
 */
void *AllocBuffer(const char *id)
{
#if defined(MUX) || defined(MUSH) || defined(MUSH3) || defined(MUX2)
    return alloc_sbuf(id);
#else
#if defined(PENN)
    return pse_malloc(MAX_ATTRIBUTE_LEN);
#else
#error No platform defined.
#endif
#endif
}

void FreeBuffer(void *buf)
{
#if defined(MUX) || defined(MUSH) || defined(MUSH3) || defined(MUX2)
    free_sbuf(buf);
#else
#if defined(PENN)
    pse_free(buf);
#else
#error No platform defined.
#endif
#endif
}

/*
 * Platform independant function to check the validity of an object.
 */
int ValidObject(dbref obj)
{
#if defined(MUX) || defined(MUSH) || defined(MUSH3) || defined(MUX2)
    return ((obj >= 0) && (obj < mudstate.db_top) &&
	    ((db[obj].flags & TYPE_MASK) < NOTYPE)) ? 1 : 0;
#else
#if defined(PENN)
    return (GoodObject(obj) && !IsGarbage(obj)) ? 1 : 0;
#else
#error No platform defined.
#endif
#endif
}

MUDFUNCTION(fun_spacecall)
{
    MUDPARAM mp;
    
#if defined(MUX) || defined(MUSH3) || defined(MUX2)
    mp.buff = buff;
    mp.bufc = bufc;
    spacecall((void*) &mp, player, cause, fargs, nfargs);
#else
#if defined(MUSH)
    mp.buff = buff;
    spacecall((void*) &mp, player, cause, fargs, nfargs);
#else
#if defined(PENN)
    mp.buff = buff;
    mp.bufc = bufc;
    mp.pe_info = pe_info;
    spacecall((void*) &mp, executor, enactor, args, nargs);
#else
#error No platform defined.
#endif
#endif
#endif
}

extern void WriteMudFunctionBuffer(void *mud, const char *str)
{
    MUDPARAM *mp = (MUDPARAM*) mud;

#if defined(MUX) || defined(MUSH3) || defined(PENN) || defined(MUX2)
    safe_str(str, mp->buff, mp->bufc);
#else
#if defined(MUSH)
    strcpy(mp->buff, str);
#else
#error No platform defined.
#endif
#endif
}

/*
 * Format up the output into the return buffer. It would be very nice if
 * there was a way to avoid vsprintf, but few platforms define a
 * vsnprintf, so we just allocate a big buffer, and cross our fingers in
 * the knowledge that 20000 is much bigger that MAX_ATTRIBUTE_LEN.
 */
extern void FWriteMudFunctionBuffer(void *mud, const char *fmt, ...)
{
    char *buff;
    MUDPARAM *mp = (MUDPARAM*) mud;
    va_list ap;

#ifdef vsnprintf
    buff = (char *) pse_malloc(MAX_ATTRIBUTE_LEN);
    va_start(ap, fmt);
    vsnprintf(buff, MAX_ATTRIBUTE_LEN, fmt, ap);
#else
    buff = (char *) pse_malloc(MAX_ATTRIBUTE_LEN * 8);
    va_start(ap, fmt);
    vsprintf(buff, fmt, ap);
#endif
    va_end(ap);
    buff[MAX_ATTRIBUTE_LEN-1] = '\0';

#if defined(MUX) || defined(PENN) || defined(MUSH3) || defined(MUX2)
    safe_str(buff, mp->buff, mp->bufc);
#else
#if defined(MUSH)
    strcpy(mp->buff, buff);
#else
#error No platform defined.
#endif
#endif

    pse_free(buff);
}

void getObjectName(char *buff, dbref obj, int maxlen)
{
#if defined(MUX) || defined(MUSH) || defined(MUSH3) || defined(MUX2)
    getAttrByNumberLen(buff, obj, A_NAME, maxlen);
#else
#if defined(PENN)
    strncpy(buff, Name(obj), maxlen);
    buff[maxlen-1] = '\0';
#else
#error No platform defined.
#endif
#endif
}

dbref parseDbref(const char *s)
{
    const char *p;
    int x;
    
    if (strlen(s) > 1) {
	for (p = s+1; *p; p++) {
	    if (!isdigit(*p))
		return PSE_NOTHING;
	}
    }
    else
	return PSE_NOTHING;
    
    x = atoi(s+1);
    return ((x >= 0) ? x : PSE_NOTHING);
}

dbref getLocation(dbref obj)
{
    return (db[obj].location);
}

dbref getDebugCharacter()
{
    /* Since 0.9.10, we use a variable set by spacecall */
    return debug_char;
}

/*
 * Return 1 if the object is permitted to use spacecalls.
 * GOD must be permitted spacecalls, as messages and events
 * are executed in that context.
 */
int isSpace(dbref obj)
{
    return (Space(obj) || (obj == GOD)) ? 1 : 0;
}

/* 
 * pse_malloc, pse_calloc, pse_free, pse_realloc, pse_strdup:
 * Safe memory allocation/deallocation routines.
 */

#define CBUFF_ENTRIES      10
#define CBUFF_SIZE         MAX_ATTRIBUTE_LEN
#define CBUFF_ENTRYLEN     (CBUFF_SIZE + 8)
#define CBUFF_FREE         0x0001
#define CBUFF_BUSY         0x0002

/*
 * Must use char not void, as Win32 doesn't like pointer arithmetic on void
 * as it's a gcc supported extension to the standard. That, or all the grief
 * John was having is a bug somewhere in the caching that Win32 exposes...
 */
static char *cb_buff = NULL;
static int cb_flags[CBUFF_ENTRIES];
static int cb_free = CBUFF_ENTRIES;
static int cb_next = 0;

void *pse_malloc(int size)
{
    char *ptr;
    int i;
    
    ptr = NULL;
    
    /* If we have a free cache slot, and the attribute is the right size */
    if (((size == MAX_ATTRIBUTE_LEN) || (size == SMALL_BUF_SIZE)) &&
	(cb_free > 0)) {
	
	/* Allocate cache ram if we haven't already */
	if (cb_buff == NULL) {
	    cb_buff = (char *) malloc(CBUFF_ENTRYLEN * CBUFF_ENTRIES);
	    assert(cb_buff != NULL);

	    cb_free = CBUFF_ENTRIES;
	    cb_next = 0;
	    for (i = 0; i < CBUFF_ENTRIES; i ++)
		cb_flags[i] = CBUFF_FREE;
	}
	
	/* Find the next free entry, starting at cb_next. */
	for (i = 0; i < CBUFF_ENTRIES; i ++)
	    if (cb_flags[(cb_next + i) % CBUFF_ENTRIES] & CBUFF_FREE) {
		cb_flags[(cb_next + i) % CBUFF_ENTRIES] = CBUFF_BUSY;
		
		/* Okay - it's a bit of ugly pointer arithmetic */
		ptr = cb_buff +
		    (CBUFF_ENTRYLEN * ((cb_next + i) % CBUFF_ENTRIES));
		cb_free--;
		cb_next = (i + 1) % CBUFF_ENTRIES;
		break;
	    }
    }
    else
	/* Allocate memory from the standard pool. Maybe this should be
	 * using the server memory allocator, just to be safe?
	 */
	ptr = (char *) malloc(size);
    
    assert(ptr != NULL);
    
    /* Just because it feels safer, make sure we return an empty string */
    *ptr = '\0';

    return(ptr);
}

void *pse_calloc(int num, int size)
{
    void *ptr;
    
    ptr = calloc(num, size);
    assert(ptr != NULL);
    
    return(ptr);
}

char *pse_strdup(const char *s)
{
    char *nstr;
    
    nstr = strdup(s);
    
    if (nstr == NULL) {
	abort();
	return(NULL);
    }
    else
	return(nstr);
}

void pse_do_free(void *ptr)
{
    char *cptr;

    /* Check for memory from the cached pool
     * (making sure the pool was allocated first)
     */

    /* Win32/VC++ pointer arithmetic on void types has been reported not to
     * work (it's not part of the ansi standard), so we do it all based on
     * char *'s - hence this cast.
     */
    cptr = (char *) ptr;

    if ((cb_buff != NULL) && (cptr >= cb_buff) &&
	(cptr < (cb_buff + CBUFF_ENTRIES*CBUFF_ENTRYLEN))) {

	/* Sanity check for making sure the pointer is on a valid boundary.
	 * If it's not, then we were passed an invalid offset into the
         * allocated cache memory, and something is really broken.
         */
	assert(((cptr-cb_buff) % CBUFF_ENTRYLEN) == 0);

	/* Repoint the next free index to this entry - after all, we know
	 * it is available!
         */
	cb_next = (cptr-cb_buff) / CBUFF_ENTRYLEN;

	/* Increment the free buffer count, and flag the buffer as free */
	cb_free++;
	cb_flags[cb_next] = CBUFF_FREE;
    }
    else
	/* Free it from the standard pool */
	free(ptr);

    return;
}

void ResetEval()
{
#if defined(MUX) || defined(MUSH3) || defined(MUSH) || defined(MUX2)
    /* Reset the function invocation counter. */
    mudstate.func_invk_ctr = 0; 
#else
#if defined(PENN)
    /* Reset the function invocation counters. */
    global_fun_invocations = global_fun_recursions = 0;
#else
#error No platform defined.
#endif
#endif
}

/* Return the dbref of the dbref attribute on the object supplied */
dbref getDbrefByNumber(dbref obj, int attr)
{
    char *buff;
    
    buff = (char *) pse_malloc(MAX_ATTRIBUTE_LEN);
    getAttrByNumberLen(buff, obj, attr, MAX_ATTRIBUTE_LEN);
    
    obj = parseDbref(buff);
    pse_free(buff);
    
    return obj;
}

/* Return the dbref of the dbref attribute on the object supplied */
dbref getDbrefByName(dbref obj, const char *attrname)
{
    char *buff;
    
    buff = (char *) pse_malloc(MAX_ATTRIBUTE_LEN);
    getAttrByNameLen(buff, obj, attrname, MAX_ATTRIBUTE_LEN);
    
    obj = parseDbref(buff);
    pse_free(buff);
    
    return obj;
}
